#=======================================================================
# miniSIPServer voice mail service script
#
# Copyright (C) MYVOIPAPP,Inc. All Rights Reserved.
#
# author: Gilson
#
# History:
# 2014-02-15 Migrated from MSS core.
#=======================================================================

import os
from mss_datatype import *
from mss_service_basic import *

# Voice mail services
EVM_TYPE_UNKNOWN    = 0
EVM_TYPE_RECORD_VM  = 1
EVM_TYPE_HEAR_VM    = 2
EVM_TYPE_RECORD_GREETING = 3
EVM_TYPE_HEAR_GREETING   = 4

# service states
EVM_STATE_IDLE = 0
EVM_STATE_RECORD_VM_NOTICE  = 1
EVM_STATE_RECORD_VM         = 2
EVM_STATE_HEAR_GREETING = 3
EVM_STATE_HEAR_VM_PLAY          = 4
EVM_STATE_HEAR_VM_WAIT_COMMAND  = 5
EVM_STATE_HEAR_VM_FINAL_WARNING = 6

# announcement ID
VM_ANN_ID_WELCOME = 0x02080004
VM_ANN_ID_BEEP    = 0x01080000 # just a beep tone
VM_ANN_ID_NO_VOICEMAIL = 0x02080005
VM_ANN_ID_WAIT_COMMAND = 0x02080003

# digit maps
VM_FIRST_DIGIT_TIMEOUT = 20
VM_INTER_DIGIT_TIMEOUT = 4

# timer values (seconds)
VM_TIMER_VAL_PLAY_ANN = 60
VM_TIMER_RECORD       = 30 # MSS core can only store 30 seconds data defined by MAX_ANN_RECD_SIZE


# noinspection PyUnusedLocal
class MSS_Service(MSS_Basic_Service):

    def __init__(self):
        MSS_Basic_Service.__init__(self)

        self.vmMode = self.getSysVarVoiceMailMode()
        self.pfxRecordVM = self.getSysVarVoiceMailPrefixRecord()
        self.pfxHearVM = self.getSysVarVoiceMailPrefixHearVM()
        self.pfxRecordGreeting = self.getSysVarVoiceMailPrefixRecordGreeting()
        self.pfxHearGreeting = self.getSysVarVoiceMailPrefixHearGreeting()

        self.vmService = EVM_TYPE_UNKNOWN
        self.vmWelcomeAnnID = VM_ANN_ID_WELCOME

        self.usrName = ''
        self.vmTotal = 0
        self.vmUnread = 0
        self.vmFileList = []
        self.vmFileIndex = 0 # which vm file is to be played
        self.vmFileName = '' # current vm file name

        self.beginRecordInd = MSS_PY_NO
        return


    def DestroyProc(self):  #Destroy process
        self.Trace("DestroyProc")
        self.SendReleaseCall()

        if self.vmService == EVM_TYPE_RECORD_VM:
            if MSS_PY_NO == self.beginRecordInd:
                return

            if self.vmMode == 0: # send vm through email
                self.Trace("\t add VM to daemon")
                self.sendVMSInfo()
            else:
                self.Trace("\t prepare to send MWI.")
                self.getVMUnreadTotal()
                self.sendVMSMWI()

        return


    def MainProc(self, currMsg): #main function
        try:
            if self.state == EVM_STATE_IDLE:
                self.onIdleProc()

            elif self.state == EVM_STATE_RECORD_VM_NOTICE:
                self.onVMAnnNoticeProc(currMsg)

            elif self.state == EVM_STATE_RECORD_VM:
                self.onRecordVMProc()

            elif self.state == EVM_STATE_HEAR_VM_PLAY:
                self.onPlayingVoiceMail(currMsg)

            elif self.state == EVM_STATE_HEAR_VM_WAIT_COMMAND:
                self.onWaitCommand(currMsg)

            elif self.state == EVM_STATE_HEAR_GREETING:
                self.EndService()

            else:
                self.Trace("unknown state: %d" % self.state)
                self.EndService()

        except:
            self.HandleExcept()
        return


    ####################################################################
    #
    # State function
    #
    ####################################################################
    def onIdleProc(self):
        self.Trace("onIdleProc: voice mail")
        result = self.detectServiceType()
        if result < 0:
            self.Trace("\t fail to detect VM type. errCode=%d" % result)
            self.EndService()
            return

        # check IDP parameters
        result = self.checkIDP()
        if result < 0:
            self.Trace("\t invalid IDP parameter. errCode=%d" % result)
            self.EndService()
            return

        # get greeting ann id
        self.vmWelcomeAnnID = self.getLocalUserVoiceMailWelcomeAnnID()
        if self.vmWelcomeAnnID <= 0:
            self.Trace("\t fail to get greeting ann ID (%d)." % self.vmWelcomeAnnID)
            self.EndService()
            return

        self.usrName = self.callerNbr

        self.SendCTR(CONTROL_LEGID)

        # process different voice-mail procedures
        if self.vmService == EVM_TYPE_RECORD_VM:
            self.prepareRecordVoiceMail()

        elif self.vmService == EVM_TYPE_HEAR_VM:
            self.beginHearVoiceMail()

        elif self.vmService == EVM_TYPE_HEAR_GREETING:
            self.hearGreetingVoice()

        elif self.vmService == EVM_TYPE_RECORD_GREETING:
            self.recordGreetingVoice()

        else:
            self.Trace("\t unknown vm service? ")
            self.EndService()
            return

        return


    def onVMAnnNoticeProc(self, currMsg):
        self.Trace("onVMAnnNoticeProc")
        if currMsg != ESCP_EVT_SRR:
            self.Trace("\t unnecessary to continue service.")
            self.EndService()
            return

        if self.vmService != EVM_TYPE_RECORD_VM \
            and self.vmService != EVM_TYPE_RECORD_GREETING:
            self.Trace("\n cannot record vm for invalid service.")
            self.EndService()
            return

        self.sendRecordVM()
        self.StartTimer(VM_TIMER_RECORD+2)
        self.EnterState(EVM_STATE_RECORD_VM)
        return


    def onRecordVMProc(self):
        self.Trace("onRecordVMProc")
        self.EndService()
        return


    def onPlayingVoiceMail(self, currMsg):
        self.Trace("onPlayingVoiceMail")
        if currMsg != ESCP_EVT_SRR:
            self.Trace("\n unnecessary to continue playing vm.")
            self.EndService()
            return

        # current voice mail has been played, now wait for customer's command
        self.sendPNCWaitCommand()

        self.StartTimer(VM_TIMER_VAL_PLAY_ANN)
        self.EnterState(EVM_STATE_HEAR_VM_WAIT_COMMAND)
        return


    def onWaitCommand(self, currMsg):
        self.Trace("onWaitCommand")
        self.StopTimer()
        if currMsg == ESCP_EVT_PNC_RESULT:
            self.recvCommand()

        elif currMsg == ESCP_EVT_TIMEOUT:
            self.prepareNextVoicemail()

        elif currMsg == ESCP_EVT_SRF_ERR:
            self.recvCommandError()

        else:
            self.Trace("\t unnecessary message. stop service.")
            self.EndService()

        return


    def prepareRecordVoiceMail(self):
        self.Trace("prepareRecordVoiceMail ")

        self.usrName = self.redirNbr
        self.Trace("\t user name is '%s'" % self.usrName)

        self.playGreetingAndBeep()
        self.beginRecordInd = MSS_PY_YES

        self.StartTimer(VM_TIMER_VAL_PLAY_ANN)
        self.EnterState(EVM_STATE_RECORD_VM_NOTICE)
        return


    def beginHearVoiceMail(self):
        self.Trace("beginHearVoiceMail")
        self.getVMUnreadTotal() # get vm file counts and all file names
        if self.vmTotal == 0:
            self.Trace("\t no voice mail")
            self.playNoVoiceMail()
            self.StartTimer(VM_TIMER_VAL_PLAY_ANN)
            self.EnterState(EVM_STATE_HEAR_VM_FINAL_WARNING)
            return

        self.vmFileName  = self.getCurrVMFileName()
        self.playVoicemail(self.vmFileName)

        self.StartTimer(VM_TIMER_VAL_PLAY_ANN)
        self.EnterState(EVM_STATE_HEAR_VM_PLAY)
        return


    def hearGreetingVoice(self):
        self.Trace("hearGreetingVoice")

        self.playGreeting()

        self.StartTimer(VM_TIMER_VAL_PLAY_ANN)
        self.EnterState(EVM_STATE_HEAR_GREETING)
        return


    def recordGreetingVoice(self):
        self.Trace("recordGreetingVoice")

        self.playGreetingAndBeep()
        self.beginRecordInd = MSS_PY_YES

        self.StartTimer(VM_TIMER_VAL_PLAY_ANN)
        self.EnterState(EVM_STATE_RECORD_VM_NOTICE)
        return


    ####################################################################
    #
    # Event functions
    #
    ####################################################################
    def recvCommand(self):
        self.Trace("\t recvCommand")
        if self.pncResult == '1':
            self.prepareNextVoicemail()

        elif self.pncResult == '0':
            self.delCurrVoiceMail()
            self.prepareNextVoicemail()

        else:
            self.prepareNextVoicemail()

        return


    def recvCommandError(self):
        self.Trace("\t recvCommandError")
        if self.srfError != ESRF_ERR_FIRST_DIGIT_TIMEOUT:
            self.Trace("\t\t MG reports error(%d), stop service." % self.srfError)
            self.EndService()
            return

        self.prepareNextVoicemail()
        return

    def prepareNextVoicemail(self):
        self.Trace("\t prepareNextVoicemail")
        if self.vmUnread > 0:
            self.vmUnread -= 1
        else:
            self.playNoVoiceMail()
            self.StartTimer(VM_TIMER_RECORD)
            self.EnterState(EVM_STATE_HEAR_VM_FINAL_WARNING)
            return

        self.sendVMSMWI()

        self.vmFileName = self.getCurrVMFileName()
        self.playVoicemail(self.vmFileName)

        self.StartTimer(VM_TIMER_VAL_PLAY_ANN)
        self.EnterState(EVM_STATE_HEAR_VM_PLAY)
        return


    ####################################################################
    #
    # Common function
    #
    ####################################################################
    def detectServiceType(self):
        self.Trace("\t detectServiceType")
        calledNbr = self.calledNbr
        if calledNbr.startswith(self.pfxRecordVM):
            self.vmService = EVM_TYPE_RECORD_VM

        elif calledNbr.startswith(self.pfxHearVM):
            self.vmService = EVM_TYPE_HEAR_VM

        elif calledNbr.startswith(self.pfxRecordGreeting):
            self.vmService = EVM_TYPE_RECORD_GREETING

        elif calledNbr.startswith(self.pfxHearGreeting):
            self.vmService = EVM_TYPE_HEAR_GREETING

        else:
            return -1
        return 0


    def checkIDP(self):
        self.Trace("\t checkIDP")
        if self.vmService == EVM_TYPE_RECORD_VM:
            if len(self.redirNbr) == 0:
                return -1

        return 0


    def playGreeting(self):
        self.Trace("\t playGreeting")
        pa_arg = PTpa_arg()
        PTpa_arg_init(pa_arg)

        pa_arg.infoSend.annId.annType = ESRF_INFO_ANN_ID
        pa_arg.infoSend.annId.var.annId = self.vmWelcomeAnnID
        pa_arg.infoSend.numOfRepeat = 1
        pa_arg.infoSend.duration    = VM_TIMER_VAL_PLAY_ANN + 10

        self.SendPA(CONTROL_LEGID, pa_arg)
        return

    def playGreetingAndBeep(self):
        self.Trace("\t playGreetingAndBeep")

        pa_arg = PTpa_arg()
        PTpa_arg_init(pa_arg)

        pa_arg.infoSend.annId.annType = ESRF_INFO_ANN_ID
        pa_arg.infoSend.annId.var.annId = self.vmWelcomeAnnID
        pa_arg.infoSend.numOfRepeat = 1
        pa_arg.infoSend.duration    = VM_TIMER_VAL_PLAY_ANN + 10

        pa_arg.infoSend.infoVar.varType = ESRF_INFO_VAR_MULTI_ID
        pa_arg.infoSend.infoVar.var.multiAnn.cnt = 1
        pa_arg.infoSend.infoVar.var.multiAnn.annID[0] = VM_ANN_ID_BEEP

        self.SendPA(CONTROL_LEGID, pa_arg)
        return


    def playNoVoiceMail(self):
        self.Trace("\t playNoVoiceMail")
        pa_arg = PTpa_arg()
        PTpa_arg_init(pa_arg)

        pa_arg.infoSend.annId.annType = ESRF_INFO_ANN_ID
        pa_arg.infoSend.annId.var.annId = VM_ANN_ID_NO_VOICEMAIL
        pa_arg.infoSend.numOfRepeat = 1
        pa_arg.infoSend.duration    = VM_TIMER_VAL_PLAY_ANN + 10

        self.SendPA(CONTROL_LEGID, pa_arg)
        return


    def playVoicemail(self, currVMFileName):
        self.Trace("\t playVoicemail")

        vmFullFileName = self.getCurrVMFullFileName(currVMFileName)
        self.Trace("\t\t VM full file name is'%s'" % vmFullFileName)

        pa_arg = PTpa_arg()
        PTpa_arg_init(pa_arg)

        pa_arg.infoSend.annId.annType = ESRF_INFO_ANN_ID
        pa_arg.infoSend.annId.var.annId = 0 # must be null
        pa_arg.infoSend.numOfRepeat = 1
        pa_arg.infoSend.duration    = VM_TIMER_VAL_PLAY_ANN + 10

        pa_arg.infoSend.infoVar.varType = ESRF_INFO_VAR_ANN_FILE
        pa_arg.infoSend.infoVar.var.annFileName = vmFullFileName

        self.SendPA(CONTROL_LEGID, pa_arg)
        return


    def sendPNCWaitCommand(self):
        self.Trace("\t sendPNCWaitCommand")
        pnc_arg = PTpnc_arg()
        PTpnc_arg_int(pnc_arg)

        pnc_arg.infoSend.annId.annType = ESRF_INFO_ANN_ID
        pnc_arg.infoSend.annId.var.annId = VM_ANN_ID_WAIT_COMMAND
        pnc_arg.infoSend.numOfRepeat = 1

        pnc_arg.digitMap.maxNumOfDigits = 1 # just collect 1 digit
        pnc_arg.digitMap.minNumOfDigits = 1
        pnc_arg.digitMap.firstDigitTimeout = VM_FIRST_DIGIT_TIMEOUT
        pnc_arg.digitMap.intervalDigitTimeout =  VM_INTER_DIGIT_TIMEOUT

        self.SendPNCWithArg(CONTROL_LEGID, pnc_arg)
        return


    def sendRecordVM(self):
        self.Trace("\t sendRecordVM")
        recordAnnArg = PTrecordann_arg()
        PTrecordann_arg_init(recordAnnArg)

        if self.vmService == EVM_TYPE_RECORD_VM:
            recordAnnArg.type = ERECORD_ANN_TYPE_VM

        elif self.vmService == EVM_TYPE_RECORD_GREETING:
            recordAnnArg.type = ERECORD_ANN_TYPE_GREETING

        else:
            return -1

        recordAnnArg.callerNbr = self.callerNbr
        recordAnnArg.subNbr = self.usrName
        self.SendRecordANN(CONTROL_LEGID, recordAnnArg)
        return 0


    def getVMUnreadTotal(self):
        self.Trace("\t getVMUnreadTotal")
        self.vmTotal = self.getAllVMFiles()
        self.vmUnread = self.vmTotal
        self.Trace("\t\t total = unread = %d" % self.vmTotal)
        return


    def getSubVMDir(self):
        exeDir = self.getExeDirName()
        vmDir = exeDir +"/vms"
        subVmDir = vmDir + "/"+self.usrName
        return subVmDir

    def getAllVMFiles(self):
        subVmDir = self.getSubVMDir()
        self.vmFileList = os.listdir(subVmDir)
        fileCnt = len(self.vmFileList)
        return fileCnt


    def getCurrVMFileName(self):
        fileName = self.vmFileList[self.vmFileIndex]
        self.Trace("\t current VM file name is '%s'" % fileName)
        self.vmFileIndex = (self.vmFileIndex + 1) % self.vmTotal  # prepare next vm file
        return fileName


    def getCurrVMFullFileName(self, vmfileName):
        subVmDir = self.getSubVMDir()
        vmFullFileName = subVmDir+"/"+vmfileName
        return vmFullFileName


    def delCurrVoiceMail(self):
        self.Trace("\t delCurrVoiceMail")
        vmFullFileName = self.getCurrVMFullFileName(self.vmFileName)
        os.remove(vmFullFileName)

        self.vmFileList.remove(self.vmFileName)
        self.vmTotal = len(self.vmFileList)
        if self.vmTotal == 0:
            self.Trace("\t\t no VM now.")
            return -1

        self.vmFileIndex %= self.vmTotal
        return 0

    ###################################################################
    #
    # Core API functions
    #
    ###################################################################
    def sendVMSInfo(self):
        self.Trace("\t sendVMSInfo")
        self.ifDll.SendVoiceMailInfo(self.scpFsmId, self.usrName)
        return


    def sendVMSMWI(self):
        self.Trace("\t sendVMSMWI")
        self.Trace("\t\t total=%d, unread=%d" % (self.vmTotal, self.vmUnread))
        self.ifDll.SendVoiceMailMWI(self.scpFsmId, self.usrName, self.vmTotal, self.vmUnread)
        return