#IVR-XML service

import os
from xml.dom import minidom
import urllib2

from mss_datatype import *
from mss_service_basic import *

def findFirstChildNodeByName(parent, childName):
    for node in parent.childNodes:
        if node.nodeType == node.ELEMENT_NODE and node.localName == childName:
            return node
    return None

# By default, node data is unicode
def getNodeData(node):
    return node.childNodes[0].data

def getNodeAttr(node, name):
    return node.getAttribute(name)

# MSS inner IVR-XML services.
# key should be same with Exml_services defined in mss core.
MSS_INNER_IVRXML = {1: "aa.xml"}


# IVR action methods
IVR_METHOD_PNC      = "playandwaitinput"
IVR_METHOD_PA       = "playannouncement"
IVR_METHOD_CALLTO   = "callto"
IVR_METHOD_INPUT    = "input"
IVR_METHOD_RELCALL  = "releasecall"
IVR_METHOD_GETURL   = "geturl"
IVR_METHOD_GETURL_RESULT = "geturlresult"


#FSM service state
EIVR_IDLE = 0
EIVR_INTERACTION = 1
EIVR_WAIT_ANSWER = 2

#default timer (seconds)
IVR_TIMER_INTERACTION = 120
IVR_TIMER_WAIT_ANSWER = 120

#noinspection PyBroadException
class MSS_Service(MSS_Basic_Service):

    def __init__(self):
        MSS_Basic_Service.__init__(self)
        self.dialPlan = ""
        self.ivrInd = MSS_PY_NO
        self.ivrxmlFileName = ""
        self.geturlResult = ""

        self.previousActionIndex = 0
        self.previousActionMethod = ""
        self.currActionIndex = 0
        self.currActionMethod = ""
        self.actionDict = {}  # use action 'name' as key, store parsed actions
        self.actionNameList = []  # use index to search action name
        return


    def MainProc(self, currMsg):  # main function
        self.StopTimer()
        try:
            if EIVR_IDLE == self.state:
                self.onIdleProc()

            elif EIVR_INTERACTION == self.state:
                self.onInteractionProc(currMsg)

            elif EIVR_WAIT_ANSWER == self.state:
                self.onWaitAnswerProc(currMsg)

            else:
                self.Trace("unknown state %d" % self.state)
                self.EndService()

        except:
            self.HandleExcept()

        return

    def DestroyProc(self):  #Destroy process
        self.Trace("\t DestroyProc")
        return
        
    #========================
    #
    # state procedures
    #
    #========================
    def onIdleProc(self):
        self.Trace("IVR-XML: onIdleProc")
        self.initData()

        result = self.getCustIVRXML()
        if result < 0:
            self.Trace("\t cannot find IVR-XML file.")
            self.exception()
            return

        result = self.parseXMLFile()
        if result < 0:
            self.Trace("\t fail to parser actions. errCode=%d" % result)
            self.exception()
            return

        self.monitorCall()
        self.prepareIVR()

        result = self.procCurrAction()
        if result != MSS_ERR_OK:
            self.exception()
            return
        return


    def onInteractionProc(self, currMsg):
        self.Trace("onInteractionProc")
        if currMsg == ESCP_EVT_PNC_RESULT:
            self.srfError = ESRF_ERR_OK
            result = self.interactionPnCResult()

        elif currMsg == ESCP_EVT_SRF_ERR:
            self.pncResult=""
            result = self.interactionSRFError()

        elif currMsg == ESCP_EVT_SRR:
            result = self.interactionSRR()

        else:
            self.exception()
            return

        if MSS_ERR_OK != result:
            self.exception()
        return


    def onWaitAnswerProc(self, currMsg):
        self.Trace("onWaitAnswerProc")

        if currMsg == ESCP_EVT_ERB:
            if DP_O_ANSWER == self.edpEvent:   # called party answers the call
                self.stopIVR()
                self.SendContinue()
                self.EndService()
                return MSS_ERR_OK

        self.exception()
        return MSS_ERR_OK


    def interactionPnCResult(self):
        self.Trace("\t interactionPnCResult : %s" % self.pncResult)

        currActionName = self.actionNameList[self.currActionIndex]
        currAction = self.actionDict[currActionName]
        currMethod = currAction["method"]
        if currMethod != IVR_METHOD_INPUT:
            self.Trace("\t\t current action is not 'input'")
            return MSS_ERR_INVALID_OPR

        # save PnC result into current 'input' action
        currAction['result'] = self.pncResult
        result = self.procCurrAction()
        if result != MSS_ERR_OK:
            return MSS_ERR_FAIL
        return MSS_ERR_OK


    def interactionSRFError(self):
        self.Trace("\t interactionSRFError")

        result = self.procCurrAction()
        if result != MSS_ERR_OK:
            return MSS_ERR_FAIL
        return MSS_ERR_OK


    def interactionSRR(self):
        self.Trace("\t interactionSRR")
        self.prepareNextAction()

        result = self.procCurrAction()
        if result != MSS_ERR_OK:
            return MSS_ERR_FAIL
        return MSS_ERR_OK


    #========================
    #
    # common procedures
    #
    #========================
    def initData(self):
        self.dialPlan = self.origDialPlan
        return


    def exception(self):
        self.Trace("\t exception")
        self.SendReleaseCall()
        self.EndService()
        return


    def getCustIVRXML(self):
        self.Trace("\t getCustIVRXML")
        if 0 != self.ivrXMLServiceID:
            self.ivrxmlFileName = MSS_INNER_IVRXML[self.ivrXMLServiceID]

        result = self.dbGetCustIVRXML(self.dialPlan, self.calledNbr)
        if result < 0:
            self.dbGetCustIVRXML(MSS_DEFAULT_DIAL_PLAN, self.calledNbr)

        if "" == self.ivrxmlFileName:
            return -1
        else:
            return 0


    def monitorCall(self):
        self.Trace("\t monitorCall")
        self.SendRRBE(CALLED_LEGID, DP_O_ANSWER, DP_MON_INTERUPT)
        return


    def prepareIVR(self):
        self.Trace("\t prepareIVR")
        if MSS_PY_NO == self.ivrInd:
            self.SendCTR(CALLER_LEGID)
            self.ivrInd = MSS_PY_YES
        return


    def stopIVR(self):
        self.Trace("\t stopIVR")
        if MSS_PY_YES == self.ivrInd:
            self.SendDFC(CALLER_LEGID)
            self.ivrInd = MSS_PY_NO
        return


    #========================
    #
    # process actions
    #
    #========================
    def procCurrAction(self):
        self.Trace("\t procCurrAction")
        currActionName = self.actionNameList[self.currActionIndex]
        currAction = self.actionDict[currActionName]
        if currAction is None:
            self.Trace("\t\t cannot find action")
            return MSS_ERR_OBJ_NOT_FOUND

        currMethod = currAction["method"]
        self.currActionMethod = currMethod
        if currMethod == IVR_METHOD_PNC:
            return self.procActionPnC(currAction)

        elif currMethod == IVR_METHOD_INPUT:
            return self.procActionInput(currAction)

        elif currMethod == IVR_METHOD_PA:
            return self.procActionPA(currAction)

        elif currMethod == IVR_METHOD_CALLTO:
            return self.procActionCallto(currAction)

        elif currMethod == IVR_METHOD_RELCALL:
            return self.procActionRelCall(currAction)

        elif currMethod == IVR_METHOD_GETURL:
            return self.procActionGetURL(currAction)

        elif currMethod == IVR_METHOD_GETURL_RESULT:
            return self.procActionGetURLResult(currAction)

        else:
            self.Trace("\t\t unknown method: %s" % str(currMethod))

        return MSS_ERR_OK


    def procActionPnC(self, currAction):
        self.Trace("\t procActionPnC")
        pncArg = PTpnc_arg()
        PTpnc_arg_int(pncArg)

        if MSS_ERR_OK != self.convertPncArg(currAction, pncArg):
            return MSS_ERR_INVALID_OPR

        self.SendPNCWithArg(CALLER_LEGID, pncArg)

        self.prepareNextAction()
        self.StartTimer(IVR_TIMER_INTERACTION)
        self.EnterState(EIVR_INTERACTION)
        return MSS_ERR_OK


    def procActionPA(self, currAction):
        self.Trace("\t procActionPA")
        paArg = PTpa_arg()
        PTpa_arg_init(paArg)

        infoSend = currAction["playaudio"]
        if MSS_ERR_OK != self.convertInfoSend(infoSend, paArg.infoSend):
            return MSS_ERR_INVALID_PARAM

        self.SendPA(CALLER_LEGID, paArg)

        self.StartTimer(IVR_TIMER_INTERACTION)
        self.EnterState(EIVR_INTERACTION)
        return MSS_ERR_OK


    def procActionInput(self, currAction):
        self.Trace("\t procActionInput")

        nextActionName = ""
        if self.srfError == ESRF_ERR_OK:
            # in 'parseActionInput', action use digit value as key
            if "digit" in currAction:
                defaultDigitElem = currAction["digit"]
                nextActionName = defaultDigitElem["nextaction"]

            digitKey = "digit"+self.pncResult
            if digitKey in currAction:
                digitElem = currAction[digitKey]
                nextActionName = digitElem["nextaction"]

        elif self.srfError == ESRF_ERR_FIRST_DIGIT_TIMEOUT:
            if "error-first-digit-timeout" in currAction:
                nextActionName = currAction["error-first-digit-timeout"]
            else:
                self.Trace("\t\t cannot find next action in 'error-first-digit-timeout' ")
                return MSS_ERR_OBJ_NOT_FOUND

        elif self.srfError == ESRF_ERR_IMPROPER_CALLER_RESPONSE:
            if "error-improper-caller-response" in currAction:
                nextActionName = currAction["error-improper-caller-response"]
            else:
                self.Trace("\t\t cannot find next in 'error-improper-caller-response'")
                return MSS_ERR_OBJ_NOT_FOUND

        else:
            self.Trace("\t\t don't know how to process current input value.")
            return MSS_ERR_INVALID_OPR

        if len(nextActionName) == 0:
            self.Trace("\t\t cannot find next action name. Just prepare next action directly.")
            self.prepareNextAction()
        else:
            self.Trace("\t\t prepare next action '%s'" % str(nextActionName))
            self.previousActionMethod = IVR_METHOD_INPUT
            if nextActionName in self.actionDict:
                nextAction = self.actionDict[nextActionName]
                self.currActionIndex = nextAction["actionIndex"]
            else:
                self.Trace("\t\t error! cannot find next action.")
                return MSS_ERR_INVALID_OBJ

        return self.procCurrAction()


    def procActionCallto(self, currAction):
        self.Trace("\t procActionCallto")

        if "number" in currAction:
            destNbr = currAction["number"]

        elif "input" in currAction:
            inputNode = currAction["input"]
            destNbr = self.getInputResult(inputNode)
        else:
            return MSS_ERR_INVALID_OBJ

        if len(destNbr)==0:
            self.Trace("\t\t cannot get destination number")
            return MSS_ERR_OBJ_NOT_FOUND

        self.SendRingBackToneToCaller()
        self.SendConnect(str(destNbr))  # convert unicode to ascii string

        self.StartTimer(IVR_TIMER_WAIT_ANSWER)
        self.EnterState(EIVR_WAIT_ANSWER)
        return MSS_ERR_OK


    def procActionRelCall(self, currAction):
        self.Trace("\t procActionRelCall")

        if "cause" in currAction:
            causeVal = currAction["cause"]
            self.SendReleaseCall(int(causeVal));
        else:
            self.SendReleaseCall()

        self.EndService()
        return MSS_ERR_OK


    def procActionGetURL(self, currAction):
        self.Trace("\t procActionGetURL")
        fullURL = self.getFullURL(currAction)
        if len(fullURL) == 0:
            self.Trace("\t\t blank url? ")

        response = urllib2.urlopen(fullURL)
        self.geturlResult = response.read()
        self.Trace("\t\t result is '%s'" % str(self.geturlResult))

        self.prepareNextAction()
        self.procCurrAction()
        return MSS_ERR_OK


    def procActionGetURLResult(self, currAction):
        self.Trace("\t procActionGetURLResult")

        #save original geturl result
        currAction["result"] =  self.geturlResult

        # get default next action
        nextActionName = ""
        if "value" in currAction:
            resultElem = currAction["value"]
            nextActionName = resultElem["nextaction"]

        valueKey = "value"+self.geturlResult
        if valueKey in currAction:
            resultElem = currAction[valueKey]
            nextActionName = resultElem["nextaction"]

        if len(nextActionName) == 0:
            self.Trace("\t\t cannot find next action name, go to next action directly.")
            self.prepareNextAction()

        else:
            self.Trace("\t\t prepare next action '%s'" % str(nextActionName))
            self.previousActionMethod = IVR_METHOD_GETURL_RESULT
            if nextActionName in self.actionDict:
                nextAction = self.actionDict[nextActionName]
                self.currActionIndex = nextAction["actionIndex"]
            else:
                self.Trace("\t\t error! cannot find next action.")
                return MSS_ERR_INVALID_OBJ

        return self.procCurrAction()


    def prepareNextAction(self):
        self.previousActionIndex = self.currActionIndex
        self.previousActionMethod = self.currActionMethod

        self.currActionIndex += 1
        return


    def convertPncArg(self, currAction, pncArg):
        if "playaudio" in currAction:
            infoSend = currAction["playaudio"]
            if MSS_ERR_OK != self.convertInfoSend(infoSend, pncArg.infoSend):
                return MSS_ERR_INVALID_PARAM

        if "minnumofdigits" in currAction:
            pncArg.digitMap.minNumOfDigits = int(currAction["minnumofdigits"])

        if "maxnumofdigits" in currAction:
            pncArg.digitMap.maxNumOfDigits = int(currAction["maxnumofdigits"])

        if "firstdigittimeout" in currAction:
            pncArg.digitMap.firstDigitTimeout = int(currAction["firstdigittimeout"])

        if "intervaldigittimeout" in currAction:
            pncArg.digitMap.intervalDigitTimeout = int(currAction["intervaldigittimeout"])

        if "enddigit" in currAction:
            pncArg.digitMap.endDigit = str(currAction["enddigit"])

        if "canceldigit" in currAction:
            pncArg.digitMap.cancelDigit = str(currAction["canceldigit"])

        return MSS_ERR_OK


    def convertInfoSend(self, infoSend, infoSendArg):
        if "id" in infoSend:
            infoSendArg.annId.annType = ESRF_INFO_ANN_ID
            infoSendArg.annId.var.annId = int(infoSend["id"],16)
        elif "text" in infoSend:
            textVal = infoSend["text"]
            infoSendArg.annId.annType = ESRF_INFO_TEXT
            infoSendArg.annId.var.text.text = str(textVal["content"])
            infoSendArg.annId.var.text.voiceName = str(textVal["voicename"])
        else:
            self.Trace("\t\t unknown message id")
            return MSS_ERR_INVALID_OBJ

        if "duration" in infoSend:
            infoSendArg.duration = int(infoSend["duration"])

        if "repeat" in infoSend:
            infoSendArg.numOfRepeat = int(infoSend["repeat"])

        return MSS_ERR_OK


    def getInputResult(self, inputNode):
        if "from" in inputNode:
            inputActName = inputNode["from"]
            return self.getInputActionResult(inputActName)
        else:
            return self.pncResult # always get current PnC result


    def getInputActionResult(self, actionName):
        inputAct = self.actionDict[actionName]
        if inputAct["method"] != IVR_METHOD_INPUT:
            return ""

        if "result" in inputAct:
            return inputAct["result"]
        else:
            return ""


    def getFullURL(self, geturlAction):
        fullURL=geturlAction["url"]

        paramCnt = geturlAction["paramCnt"]
        if paramCnt == 0:
            return fullURL

        paramIndex = 0
        paramStrs = "?"
        while paramIndex < paramCnt:
            key = "param" + str(paramIndex)
            param = geturlAction[key]
            valueName = param["name"]
            value = param["value"]
            valueType = value["type"]
            if valueType == "string":
                valueContent = value["content"]

            elif valueType == "callernbr":
                valueContent = self.callerNbr

            elif valueType == "callednbr":
                valueContent = self.calledNbr

            elif valueType == "input":
                inputNode = value["content"]
                valueContent = self.getInputResult(inputNode)

            else:
                self.Trace("\t\t unknown parameter type: %s" % str(type))
                return ""

            paramStrs += valueName+"="+valueContent
            paramIndex += 1
            if paramIndex < paramCnt:
                paramStrs += "&"

        fullURL += paramStrs
        return fullURL


    #========================
    #
    # parser functions
    #
    #========================
    def parseXMLFile(self):
        self.Trace("\t parseXMLFile")       

        ivrxmlFullName = self.getExeDirName() + "/xml/" + self.ivrxmlFileName
        self.Trace("\t\t IVR-XML file: %s" % ivrxmlFullName)
        try:
            xmlDoc = minidom.parse(ivrxmlFullName)
            xmlRoot = xmlDoc.documentElement
            actionElements = xmlRoot.getElementsByTagName("action")
        except:
            return -1

        if actionElements is None:
            return -2

        self.Trace("\t\t get %d actions" % len(actionElements))

        result = self.parseActions(actionElements)
        if result < 0:
            return -3
        else:
            return 0


    def parseActions(self, actionElements):
        self.Trace("\t parserActions");
        actionIndex = 0
        for actionElement in actionElements:
            result = self.parseAction(actionIndex, actionElement)
            if result < 0:
                return -1
            actionIndex += 1
        return 0


    def parseAction(self, actionIndex, actionElement):
        try:
            actName = getNodeAttr(actionElement, "name")
            actMethod = getNodeAttr(actionElement,"method")
        except:
            return -1

        if actName == "" or actMethod == "":
            return -2

        currAction= dict(name=actName, method=actMethod, actionIndex=actionIndex)
        if actMethod == IVR_METHOD_PNC:
            result = self.parseActionPnC(currAction, actionElement)

        elif actMethod == IVR_METHOD_INPUT:
            result = self.parseActionInput(currAction, actionElement)

        elif actMethod == IVR_METHOD_CALLTO:
            result = self.parseActionCallto(currAction, actionElement)

        elif actMethod == IVR_METHOD_RELCALL:
            result = self. parseActionRelCall(currAction, actionElement)

        elif actMethod == IVR_METHOD_PA:
            result = self.parseActionPA(currAction, actionElement)

        elif actMethod == IVR_METHOD_GETURL:
            result = self.parseActionGetURL(currAction, actionElement)

        elif actMethod == IVR_METHOD_GETURL_RESULT:
            result = self.parseActionGetURLResult(currAction, actionElement)

        else:
            self.Trace("\t unknown method: %s" % str(actMethod))
            return -3

        if result < 0:
            self.Trace("\t fail to parse action. errCode=%d" % result)
            return -4

        self.actionDict[actName] = currAction
        self.actionNameList.append(actName)
        return 0


    #========================
    #
    # parse actions
    #
    #========================
    def parseActionPnC(self, currAction, actionElement):
        self.Trace("\t parseActionPnC")
        if not actionElement.hasChildNodes():
            self.Trace("\t\t cannot find child nodes")
            return -1

        playaudioElement = findFirstChildNodeByName(actionElement, "playaudio")
        if playaudioElement is not None:
            ivrAudio = self.parseElementAudio(playaudioElement)
            if len(ivrAudio)>0:
                currAction["playaudio"] = ivrAudio

        #search 'minnumofdigits'
        minNode = findFirstChildNodeByName(actionElement,"minnumofdigits")
        if minNode is not None:
            minnumofdigits = getNodeData(minNode)
            currAction["minnumofdigits"] = minnumofdigits

        #search 'maxnumofdigits'
        maxNode = findFirstChildNodeByName(actionElement, "maxnumofdigits")
        if maxNode is not None:
            maxnumofdigits = getNodeData(maxNode)
            currAction["maxnumofdigits"] = maxnumofdigits

        #search 'firstdigittimeout'
        firstDigitNode = findFirstChildNodeByName(actionElement, "firstdigittimeout")
        if firstDigitNode is None:
            firstDigitNode = findFirstChildNodeByName(actionElement, "first-digit-timeout")

        if firstDigitNode is not None:
            firstdigittimeout = getNodeData(firstDigitNode)
            currAction["firstdigittimeout"] = firstdigittimeout

        #search 'intervaldigittimeout'
        intervaldigittimeoutNode = findFirstChildNodeByName(actionElement, "intervaldigittimeout")
        if intervaldigittimeoutNode is None:
            intervaldigittimeoutNode = findFirstChildNodeByName(actionElement, "interval-digit-timeout")

        if intervaldigittimeoutNode is not None:
            intervaldigittimeout = getNodeData(intervaldigittimeoutNode)
            currAction["intervaldigittimeout"] = intervaldigittimeout

        #search 'enddigit'
        enddigitNode = findFirstChildNodeByName(actionElement, "enddigit")
        if enddigitNode is not None:
            currAction["enddigit"] = getNodeData(enddigitNode)

        #search 'canceldigit'
        canceldigitNode = findFirstChildNodeByName(actionElement, "canceldigit")
        if canceldigitNode is not None:
            currAction["canceldigit"] = getNodeData(canceldigitNode)
        return 0


    def parseActionPA(self, currAction, actionElement):
        self.Trace("\t parseActionPA")
        if not actionElement.hasChildNodes():
            self.Trace("\t\t cannot find child nodes")
            return -1

        playaudioElement = findFirstChildNodeByName(actionElement, "playaudio")
        if playaudioElement is None:
            self.Trace("\t\t cannot find 'playaudio' node.")
            return -2

        ivrAudio = self.parseElementAudio(playaudioElement)
        if len(ivrAudio)>0:
            currAction["playaudio"] = ivrAudio

        return 0


    def parseActionInput(self, currAction, actionElement):
        self.Trace("\t parseActionInput")
        if not actionElement.hasChildNodes():
            self.Trace("\t\t cannot find child nodes")
            return -1

        #search 'digit' node
        digitNodes = actionElement.getElementsByTagName("digit")
        for digitNode in digitNodes:
            digitElement = self.parseElementDigit(digitNode)
            if digitElement is None:
                self.Trace("\t\t fail to parse 'digit' element.")
                return -2
            #since there are several digit nodes, we use digit value as key in action
            if "value" in digitElement:
                digitKey = "digit" + digitElement["value"]
            else:
                digitKey = "digit"
            currAction[digitKey] = digitElement

        #search 'error-first-digit-timeout'
        errFirstDigitNode = findFirstChildNodeByName(actionElement, "error-first-digit-timeout")
        if errFirstDigitNode is not None:
            currAction["error-first-digit-timeout"] = getNodeAttr(errFirstDigitNode, "nextaction")

        #search 'error-improper-caller-response'
        errCallerNode = findFirstChildNodeByName(actionElement, "error-improper-caller-response")
        if errCallerNode is not None:
            currAction["error-improper-caller-response"] = getNodeAttr(errCallerNode, "nextaction")

        return 0


    def parseActionCallto(self, currAction, actionElement):
        self.Trace("\t parseActionCallto")
        destNode = findFirstChildNodeByName(actionElement, "destination")
        if destNode is None:
            return -1

        inputNode = findFirstChildNodeByName(destNode, "input")
        if inputNode is None:
            destNbr = getNodeData(destNode)
            currAction["number"] = destNbr

        else:
            inputElem = self.parseElementInput(inputNode)
            if inputElem is None:
                self.Trace("\t\t fail to parse 'input' element")
                return -2
            currAction['input'] = inputElem

        return 0


    def parseActionRelCall(self, currAction, actionElement):
        self.Trace("\t parseActionRelCall")
        causeNode = findFirstChildNodeByName(actionElement, "cause")
        if causeNode is not None:
            currAction["cause"] = getNodeData(causeNode)

        return 0


    def parseActionGetURL(self, currAction, actionElement):
        self.Trace("\t parseActionGetURL")
        urlNode = findFirstChildNodeByName(actionElement, "url")
        if urlNode is None:
            self.Trace("\t\t cannot find 'url' node")
            return -1
        currAction['url'] = getNodeData(urlNode)

        paramNodes = actionElement.getElementsByTagName("parameter")
        paramCnt = len(paramNodes)
        currAction['paramCnt'] = paramCnt

        paramIndex = 0
        for paramNode in paramNodes:
            paramElem = self.parseElementParameter(paramNode)
            if paramElem is None:
                self.Trace("\t\t fail to parse 'parameter' of geturl action")
                return -2
            key = "param"+str(paramIndex) # param0, param1, param2, ...
            currAction[key] = paramElem
            paramIndex += 1
        return 0


    def parseActionGetURLResult(self, currAction, actionElement):
        self.Trace("\t parseActionGetURLResult")
        resultNodes = actionElement.getElementsByTagName("result")

        for resultNode in resultNodes:
            resultElem = self.parseElementResult(resultNode)
            if resultElem is None:
                self.Trace("\t\t fail to parse 'result' element")
                return -1

            resultKey = "value"
            if "value" in resultElem:
                resultKey += resultElem["value"]

            currAction[resultKey] = resultElem

        return 0

    #========================
    #
    # parse elements
    #
    #========================
    def parseElementAudio(self, playaudioElement):
        ivrAudio={}
        idNode = findFirstChildNodeByName(playaudioElement, "id")
        if idNode is not None:
            ivrAudio["id"]=getNodeData(idNode)
            self.Trace("\t\t id=%s" % str(ivrAudio["id"]))
        else:
            textNode = findFirstChildNodeByName(playaudioElement, "text")
            if textNode is None:
                return ivrAudio # unnecessary to continue since it is not 'id' and 'text'
            textXML = self.parseElementText(textNode)
            ivrAudio["text"] = textXML

        #search 'duration' node
        durationNode = findFirstChildNodeByName(playaudioElement, "duration")
        if durationNode is not None:
            ivrAudio["duration"] = getNodeData(durationNode)

        #search 'repeat' node
        repeatNode = findFirstChildNodeByName(playaudioElement, "repeat")
        if repeatNode is not None:
            ivrAudio["repeat"] = getNodeData(repeatNode)
        return ivrAudio


    def parseElementText(self, textNode):
        textXML={}
        textInfo = getNodeData(textNode)
        textXML["content"] = textInfo

        voiceName = getNodeAttr(textNode, "voicename")
        if len(voiceName)>0:
            textXML["voicename"]=voiceName

        return textXML


    def parseElementDigit(self, digitNode):
        digitElement={}

        #search 'nextaction'
        nextaction = getNodeAttr(digitNode, "nextaction")
        if len(nextaction) == 0:
            return None
        digitElement["nextaction"]=nextaction

        #search 'value'
        value = getNodeAttr(digitNode, "value")
        if len(value) > 0 :
            digitElement["value"] = value

        return digitElement


    def parseElementInput(self, inputNode):
        inputElement={}

        #search 'from'
        fromStr = getNodeAttr(inputNode, "from")
        if len(fromStr) > 0:
            inputElement["from"] = fromStr

        return inputElement


    def parseElementParameter(self, paramNode):
        #search 'name'
        nameNode = findFirstChildNodeByName(paramNode,"name")
        if nameNode is None:
            return None

        paramElement= {"name": getNodeData(nameNode)}

        #search 'value'
        valueNode = findFirstChildNodeByName(paramNode, "value")
        if valueNode is None:
            return None
        valueElement = self.parseElementValue(valueNode)
        if valueElement is not None:
            paramElement["value"] = valueElement
            return paramElement

        return None


    def parseElementValue(self, valueNode):
        valueElement={}
        if not valueNode.hasChildNodes():
            valueElement["type"] = "string"
            valueElement["content"] = getNodeData(valueNode)
            return valueElement

        callerNbrNode = findFirstChildNodeByName(valueNode, "callernbr")
        if callerNbrNode is not None:
            valueElement["type"] = "callernbr"
            return valueElement

        calledNbrNode = findFirstChildNodeByName(valueNode, "callednbr")
        if calledNbrNode is not None:
            valueElement["type"] = "callednbr"
            return valueElement

        inputNode = findFirstChildNodeByName(valueNode, "input")
        if inputNode is not None:
            inputValue = self.parseElementInput(inputNode)
            valueElement["type"] = "input"
            valueElement["content"] = inputValue
            return valueElement

        return None


    def parseElementResult(self, resultNode):
        resultElement={}

        #search 'nextaction'
        nextaction = getNodeAttr(resultNode, "nextaction")
        if len(nextaction) == 0:
            return None
        resultElement["nextaction"]=nextaction

        #search 'value'
        value = getNodeAttr(resultNode, "value")
        if len(value) > 0 :
            resultElement["value"] = value
        return resultElement


    #========================
    #
    # cacheDB procedures
    #
    #========================
    def dbGetCustIVRXML(self, dialPlan, calledNbr):
        sqlStr = "select filename from tbl_ivr_trigger where" \
                 "  dialPlan='%s' and  calledNbr='%s'" \
                 % (dialPlan, calledNbr)

        result = self.cdbQuery(sqlStr)
        if MSS_ERR_OBJ_FOUND != result:
            return -1

        self.ivrxmlFileName = self.cdbGetRowVal(0)
        return 0
