#==============================================================================
# miniSipServer calling card service script
#
# Copyright (C) MYVOIPAPP,Inc. All Rights Reserved.
# 
# author: Hong
#
# History:
# 2011-03-05 Migrated from MSS core.      
# 2011-03-12 Support full features.
# 2011-03-18 Fix a bug: if the call is not answered, it is unnecessary to set
#            CDR to MySQL database.
# 2011-04-11 If MSS cannot get the prepaid rate for the destination, MSS should
#            process it as "unavailable destination" and prompt fail tone to caller.
# 2011-08-30 Support one-time fee in prepaid rate
#==============================================================================

from datetime import *

from mss_datatype import *
from mss_dba import *
from mss_service_basic import *


#service states
ECC_IDLE              = 0
ECC_WAIT_CARD         = 1
ECC_WAIT_PWD          = 2
ECC_WAIT_DEST_NBR     = 3
ECC_PLAY_DURATION     = 4
ECC_PLAY_INVALID_CARD = 5
ECC_PLAY_BALANCE      = 6
ECC_PLAY_DEST_UNAVAIL = 7
ECC_PLAY_SELECTION    = 8
ECC_PLAY_BYE          = 9
ECC_WAIT_ANSWER       = 10
ECC_TALK              = 11

#Timer
CC_TIMER_PLAY     = 115 # Seconds
CC_TIMER_WAIT_ANS = 120 # Seconds

#default values
CC_CALLER_LEG = 1
CC_CALLED_LEG = 2

CC_FIRST_DIGIT_TIMEOUT = 60 # Seconds
CC_INTER_DIGIT_TIMEOUT = 4  # Seconds

CC_MAX_DURATION_FOR_FREE_CALL = 28800 # Seconds

CC_MAX_CARD_LEN = 32
CC_MAX_PWD_LEN  = 32

CC_DEFAULT_MATCH_TAG = "*"
CC_DEFAULT_DIAL_PLAN = "default"

CC_MAX_INPUT_CARD_CNT  = 3
CC_MAX_TRY_DESTINATION = 3

#other data structures
class Prate_info():
    def __init__(self):
        self.interval = 0
        self.tariff   = 0
        self.onetimeFee = 0
        
class MSS_Service(MSS_Basic_Service):    
    
    #========================
    #
    # core interfaces
    #
    #========================
    def MainProc(self, currMsg): #main function
        self.Trace("Calling card: MainProc")
        
        self.StopTimer()
        
        try:
            if ECC_IDLE == self.state:
                self.onIdleProc(currMsg)
            
            elif ECC_WAIT_CARD == self.state:
                self.onWaitCardProc(currMsg)
            
            elif ECC_WAIT_PWD == self.state:
                self.onWaitPwdProc(currMsg)
                
            elif ECC_PLAY_BALANCE == self.state:
                self.onPlayBalanceProc(currMsg)    
                
            elif ECC_WAIT_DEST_NBR == self.state:
                self.onWaitDestNbrProc(currMsg)
            
            elif ECC_PLAY_DURATION == self.state:
                self.onPlayDurationProc(currMsg)
                
            elif ECC_WAIT_ANSWER == self.state:
                self.onWaitAnswerProc(currMsg)
            
            elif ECC_TALK == self.state:
                self.onTalkProc(currMsg)
                
            elif ECC_PLAY_SELECTION == self.state:
                self.onPlaySelectionProc(currMsg)    
            
            elif ECC_PLAY_DEST_UNAVAIL == self.state:
                self.onPlayDestUnavailProc(currMsg)
            
            elif ECC_PLAY_INVALID_CARD == self.state:
                self.onPlayInvalidCardProc(currMsg)
            
            elif ECC_PLAY_BYE == self.state:
                self.onPlayByeProc(currMsg)
                
            else:
                self.Trace("unknown state %d" % self.state)
                self.EndService()
                
        except:
            self.HandleExcept()
            
        return
    
    
    def DestroyProc(self):  #Destroy process
        self.Trace("DestroyProc")
        
        self.releaseTime = datetime.now()
        
        self.calculateCdrFee()  
        
        self.updateBalance()
        
        self.dbSetCDR()
        
        self.SendReleaseCall()
        
        self.unlockCard()

        return
    
    def _exception(self):
        self.Trace("_exception")

        self.SendReleaseCall()
        self.EndService()
        return
    #========================
    #
    # state procedures
    #
    #========================
    def onIdleProc(self, currMsg):
        self.Trace("onIdleProc")
        
        self.initAllParams()
        
        
        self.setupTime = datetime.now()

        self.Trace("\t dial plan of basic call is '%s'" % self.dialPlan)
        
        self.prepareIVR()
        
        self.requestCard()
        
        self.StartTimer(CC_TIMER_WAIT_ANS)
        self.EnterState(ECC_WAIT_CARD)
        
        return
        
    
    def onWaitCardProc(self, currMsg):
        self.Trace("onWaitCardProc")
        
        if ESCP_EVT_PNC_RESULT == currMsg:
            self.recvCard()
            
        else:
            self.EndService()
        
        return
            
    
    def onWaitPwdProc(self, currMsg):
        self.Trace("onWaitPwdProc")
        
        if ESCP_EVT_PNC_RESULT == currMsg:
            self.recvPWD()
            
        else:
            self.EndService()
        
        return    
    
    def onPlayInvalidCardProc(self, currMsg):
        self.Trace("onPlayInvalidCardProc")
        
        if ESCP_EVT_SRR == currMsg:
            if CC_MAX_INPUT_CARD_CNT <= self.inputCardCnt:
                self.playBye()
                self.StartTimer(CC_TIMER_PLAY)
                self.EnterState(ECC_PLAY_BYE)
            
            else:
                self.requestCard()
                
                self.StartTimer(CC_TIMER_PLAY)
                self.EnterState(ECC_WAIT_CARD)            
        else:
            self.EndService()
        
        return
    
        
    def onPlayBalanceProc(self, currMsg):
        self.Trace("onPlayBalanceProc")
        
        if ESCP_EVT_SRR == currMsg:
            self.requestDestNbr()
            
            self.StartTimer(CC_TIMER_PLAY)
            self.EnterState(ECC_WAIT_DEST_NBR)
            
        else:
            self.EndService()
        
        return
    
    
    def onWaitDestNbrProc(self, currMsg):
        self.Trace("onWaitDestNbrProc")
        
        if ESCP_EVT_PNC_RESULT == currMsg:
            self.recvDestNbr()
            
        else:
            self.EndService()
        
        return    
    
    
    def onPlayDurationProc(self, currMsg):
        self.Trace("onPlayDurationProc")
        
        if ESCP_EVT_SRR == currMsg:
            self.makeCall()
            
        else:
            self.EndService()
            
        return
    
    
    def onWaitAnswerProc(self, currMsg):
        self.Trace("onWaitAnswerProc")        
        
        if ESCP_EVT_ERB == currMsg:
            if DP_O_ANSWER == self.edpEvent:   # called party answers the call             
                self.callConnected()
                
            else:         # For any other events, the call is fail                 
                self.callFail()
            
        else:
            self.EndService()
            
        return
    
    
    def onTalkProc(self, currMsg):
        self.Trace("onTalkProc")
        
        if ESCP_EVT_TIMEOUT == currMsg:
            self.prepareIVR()
            self.playBye()
            
            self.StartTimer(CC_TIMER_PLAY)
            self.EnterState(ECC_PLAY_BYE)

        elif ESCP_EVT_ERB == currMsg: # only O_DISC2 should be received    
            self.callEnd()
            
        else:    
            self.EndService()
        
        return
    
    
    def onPlaySelectionProc(self, currMsg):
        self.Trace("onPlaySelectionProc")
        
        if ESCP_EVT_PNC_RESULT == currMsg:            
            self.recvSelection()
        
        elif ESCP_EVT_SRF_ERR == currMsg:
            self.playBye()
            self.StartTimer(CC_TIMER_PLAY)
            self.EnterState(ECC_PLAY_BYE)
        
        else:
            self.EndService()
        
        return
    
        
    def onPlayDestUnavailProc(self, currMsg):
        self.Trace("onPlayDestUnavailProc")
        
        if CC_MAX_TRY_DESTINATION <= self.makeCallCnt:
            self.playBye()
            
            self.StartTimer(CC_TIMER_PLAY)
            self.EnterState(ECC_PLAY_BYE)
            
        else:
            self.playBalance()
            
            self.StartTimer(CC_TIMER_PLAY)
            self.EnterState(ECC_PLAY_BALANCE)
        
        return
    
    
    def onPlayByeProc(self, currMsg):
        self.Trace("onPlayByeProc")
        
        self.EndService()
        
        return
        
    #========================
    #
    # process messages functions
    #
    #========================
    def recvCard(self):
        self.Trace("recvCard")
        self.card = self.pncResult
        self.Trace("\t user input card: %s" % self.card)
        
        # check card valid
        if 0 == len(self.card):
            self.EndService()
            return
        
        self.inputCardCnt += 1
        
        result = self.dbGetCardInfo(self.card)
        
        # check whether another user is using this card
        if 0 <= result:
            if MSS_PY_YES == self.isCardLocked():
                self.Trace("\t current card is used by another user.")
                result = -100
            
        if 0 > result:
            self.Trace("\t fail to get card information.(err_code=%d)" % result)
            
            self.playInvalidCard()
            self.EnterState(ECC_PLAY_INVALID_CARD)
            return
        
        # lock card
        self.lockCard()
        
        # check password
        if 0 == len(self.pwd): # this card doesn't have password
        
            self.playBalance()
            
            self.StartTimer(CC_TIMER_PLAY)
            self.EnterState(ECC_PLAY_BALANCE)
        
        else:
            self.requestPassword()
            
            self.StartTimer(CC_TIMER_PLAY)    
            self.EnterState(ECC_WAIT_PWD)
        
        return
            
    """
    2014-04-03 some customers require to interrupt playing balance and input destination
               number directly, so here we use PNC operation to play balance and collect
               destination number.
    """
    def recvPWD(self):
        self.Trace("recvPWD")
        
        pwd = self.pncResult
        
        self.Trace("\t user inputs password '%s'" % pwd)
        
        # check password valid
        if 0 == len(pwd) or pwd != self.pwd:
            self.unlockCard()
            self.playInvalidCard()

            self.StartTimer(CC_TIMER_PLAY)
            self.EnterState(ECC_PLAY_INVALID_CARD)            
            return
        
        self.playBalanceAndRequestDest()
        
        self.StartTimer(CC_TIMER_PLAY)
        self.EnterState(ECC_WAIT_DEST_NBR)
        return
        
            
    def recvDestNbr(self):
        self.Trace("recvDestNbr")  
        
        self.destNbr = self.pncResult        
        self.Trace("\t user inputs destination number '%s'" % self.destNbr)
        
        destNbrLen = len(self.destNbr)
        
        if 0>= destNbrLen:
            self.EndService()
            return
        
        result = self.calculateDuration()
        if result < 0 :
            self.callFail()
            return 
        
        self.playDuration()
        
        self.StartTimer(CC_TIMER_WAIT_ANS)
        self.EnterState(ECC_PLAY_DURATION)
        
        return

    
    def recvSelection(self):
        self.Trace("recvSelection")
        
        selection = self.pncResult
        
        if selection == "1": # dial new destination
            self.dialNewDest()
            
        elif selection =="2": # dial the same destination
            self.dialSameDest()
        
        elif selection == "3": # use new calling card
            self.useNewCard()
            
        else:
            self.playBye()
            
            self.StartTimer(CC_TIMER_PLAY)
            self.EnterState(ECC_PLAY_BYE)    
            
        return
    
        
    def makeCall(self):
        self.Trace("makeCall")
        
        self.makeCallCnt += 1
        
        self.initCDRInfo()
        
        self.sendRRBEs() # request basic call events
        
        self.playWaitMusic()
        
        self.connectDestNbr()
        
        self.setupTime = datetime.now()
        
        self.StartTimer(CC_TIMER_WAIT_ANS)
        self.EnterState(ECC_WAIT_ANSWER)
        
        return
    
    
    def callConnected(self):
        self.Trace("callConnected")
        
        self.answerTime = datetime.now()
        
        self.stopIVR()
        
        self.SendContinue()
        
        self.StartTimer(self.allowDuration)
        
        self.EnterState(ECC_TALK)
        
        return
    
    
    def callFail(self):
        self.Trace("callFail")
        
        self.stopIVR() # stop waiting tone
        
        self.releaseTime = datetime.now()
        
        #self.calculateCdrFee()  
        
        #self.dbSetCDR()
        
        self.prepareIVR() # prepare IVR to play fail tone        
        self.playDestFail()
        
        self.StartTimer(CC_TIMER_PLAY)
        self.EnterState(ECC_PLAY_DEST_UNAVAIL)
                
        return
        
    
    def callEnd(self):
        self.Trace("callEnd")
        
        self.releaseTime = datetime.now()
        
        self.calculateCdrFee()
        
        self.updateBalance()
        
        self.dbSetCDR()
        
        self.prepareIVR()
        
        if CC_MAX_TRY_DESTINATION < self.makeCallCnt:
            self.playBye()
            
            self.StartTimer(CC_TIMER_PLAY)
            self.EnterState(ECC_PLAY_BYE)
                        
        else:            
            self.requestSelection()
        
            self.StartTimer(CC_TIMER_PLAY)
            self.EnterState(ECC_PLAY_SELECTION)
                  
        return
    
    
    def dialNewDest(self):
        self.Trace("dialNewDest")
        
        self.balance = self.dbGetBalance()
        
        self.initCallInfo()
        self.destNbr = ""
        
        self.playBalance()
        
        self.StartTimer(CC_TIMER_PLAY)
        self.EnterState(ECC_PLAY_BALANCE)
        return
    
    
    def dialSameDest(self):
        self.Trace("dialSameDest")
        
        self.balance = self.dbGetBalance()
        
        self.initCallInfo()
        
        result = self.calculateDuration()
        if result < 0 :
            self.callFail()
            return          
        
        self.playDuration()
        
        self.StartTimer(CC_TIMER_PLAY)
        self.EnterState(ECC_PLAY_DURATION)
        return
    
    
    def useNewCard(self):
        self.Trace("useNewCard")
        
        self.makeCallCnt = 0
        self.inputCardCnt = 0
        self.destNbr = ""

        self.unlockCard()

        self.initCallInfo()
        self.initCardInfo()
        self.initCDRInfo()

        self.requestCard()
        self.StartTimer(CC_TIMER_PLAY)
        self.EnterState(ECC_WAIT_CARD)
        return
    
    #========================
    #
    # inner functions
    #
    #========================
    def initAllParams(self):
        self.ivrInd      = MSS_PY_NO
        self.makeCallCnt = 0
        self.inputCardCnt=0
        self.destNbr     = ""
        
        self.initCallInfo()
        self.initCardInfo()
        self.initCDRInfo()
        
        return
        

    def initCallInfo(self):        
        self.dialPlan      = self.origDialPlan        
        self.allowDuration = 0

        return


    def initCardInfo(self):
        self.card        = ""
        self.pwd         = ""
        self.balance     = 0
        self.rateId      = 0
        self.rateInfo    = Prate_info()
        self.dialPlanInd = 0  
        
        self.cardLockerInd = MSS_PY_NO  
        return


    def initCDRInfo(self):
        self.setCDRInd   = MSS_PY_NO
        self.setupTime   = 0
        self.answerTime  = 0
        self.releaseTime = 0
        self.cdrDuration = 0
        self.cdrFee      = 0
        return
        
    def getLockerName(self):
        lockerName = "card=%s" % self.card
        return lockerName
        
    def isCardLocked(self):
        if 0 == len(self.card):
            return MSS_PY_NO
            
        lockerName = self.getLockerName()
        result = self.hasLocker(lockerName)
        if MSS_PY_YES == result:
            return MSS_PY_YES
        
        return MSS_PY_NO
        
    def lockCard (self):
        if MSS_PY_YES == self.cardLockerInd or 0 == len(self.card):
            return
        
        lockerName = self.getLockerName()
        self.setLocker(lockerName)
        
        self.cardLockerInd = MSS_PY_YES
        self.Trace("\t lockCard")
        return
        
    def unlockCard (self):
        if MSS_PY_NO == self.cardLockerInd or 0 == len(self.card):
            return
        
        lockerName = self.getLockerName()
        self.clrLocker(lockerName)

        self.card = ""
        self.cardLockerInd = MSS_PY_NO
        self.Trace("\t unlockCard")
        return
        
    
    def sendRRBEs(self):
        self.SendRRBE(CC_CALLED_LEG, DP_ROUTE_SELECT_FAILURE,  DP_MON_INTERUPT) 
        self.SendRRBE(CC_CALLED_LEG, DP_O_BUSY,                DP_MON_INTERUPT) 
        self.SendRRBE(CC_CALLED_LEG, DP_O_NO_ANSWER,           DP_MON_INTERUPT)
        self.SendRRBE(CC_CALLED_LEG, DP_O_DISCONNECT,          DP_MON_INTERUPT) 
        self.SendRRBE(CC_CALLED_LEG, DP_O_ANSWER,              DP_MON_INTERUPT)
        return  
        
    
    def prepareIVR(self):
        if MSS_PY_NO == self.ivrInd:
            self.SendCTR(CC_CALLER_LEG)    
            self.ivrInd = MSS_PY_YES
        return    
        
    
    def stopIVR(self):
        if MSS_PY_YES == self.ivrInd:
            self.SendDFC(CC_CALLER_LEG)
            self.ivrInd = MSS_PY_NO
            
        return
        
            
    def requestCard(self):
        self.SendPNC(CC_CALLER_LEG, 0x84080001, CC_FIRST_DIGIT_TIMEOUT, CC_INTER_DIGIT_TIMEOUT)
        return
        
                
    def requestPassword(self):
        self.SendPNC(CC_CALLER_LEG, 0x84080002, CC_FIRST_DIGIT_TIMEOUT, CC_INTER_DIGIT_TIMEOUT)
        return
        
            
    def requestDestNbr(self):
        self.SendPNC(CC_CALLER_LEG, 0x84080003, CC_FIRST_DIGIT_TIMEOUT, CC_INTER_DIGIT_TIMEOUT)
        return
    
    
    def requestSelection(self):
        pnc_arg = PTpnc_arg()
        PTpnc_arg_int(pnc_arg)
        
        pnc_arg.infoSend.annId.annType = ESRF_INFO_ANN_ID
        pnc_arg.infoSend.annId.var.annId = 0x84080009
        pnc_arg.infoSend.numOfRepeat = 1
        
        pnc_arg.digitMap.maxNumOfDigits = 1 # just collect 1 digit
        
        self.SendPNCWithArg(CC_CALLER_LEG, pnc_arg)
        return
        
            
    def playDuration(self):
        pa_arg = PTpa_arg()
        PTpa_arg_init(pa_arg)
        
        pa_arg.infoSend.annId.annType = ESRF_INFO_ANN_ID
        pa_arg.infoSend.annId.var.annId = 0x84080004
        pa_arg.infoSend.numOfRepeat = 1
        
        pa_arg.infoSend.infoVar.varType      = ESRF_INFO_VAR_DURATION
        pa_arg.infoSend.infoVar.var.duration = self.allowDuration        
        
        self.SendPA(CC_CALLER_LEG, pa_arg)
        return
        
    
    def playBalance(self):
        self.Trace("\t playBalance")
        
        pa_arg = PTpa_arg()
        PTpa_arg_init(pa_arg)
        
        pa_arg.infoSend.annId.annType = ESRF_INFO_ANN_ID
        pa_arg.infoSend.annId.var.annId = 0x84080005
        pa_arg.infoSend.numOfRepeat = 1
        pa_arg.infoSend.duration    = CC_TIMER_WAIT_ANS
        
        pa_arg.infoSend.infoVar.varType   = ESRF_INFO_VAR_PRICE
        pa_arg.infoSend.infoVar.var.price = self.balance        
        
        self.SendPA(CC_CALLER_LEG, pa_arg) 
        return


    def playBalanceAndRequestDest(self):
        self.Trace("\t playBalanceAndCollectDest")
        pnc_arg = PTpnc_arg()
        PTpnc_arg_int(pnc_arg)

        pnc_arg.infoSend.annId.annType = ESRF_INFO_ANN_ID
        pnc_arg.infoSend.annId.var.annId = 0x84080005
        pnc_arg.infoSend.numOfRepeat = 1
        pnc_arg.infoSend.duration = CC_TIMER_WAIT_ANS

        pnc_arg.infoSend.infoVar.varType   = ESRF_INFO_VAR_PRICE
        pnc_arg.infoSend.infoVar.var.price = self.balance

        pnc_arg.infoSend.infoVar2.varType   = ESRF_INFO_VAR_MULTI_ID
        pnc_arg.infoSend.infoVar2.var.multiAnn.cnt = 1
        pnc_arg.infoSend.infoVar2.var.multiAnn.annID[0] = 0x84080003

        self.SendPNCWithArg(CC_CALLER_LEG, pnc_arg)
        return
    
    def playDestFail(self):
        self.Trace("\t playDestFail")
        
        pa_arg = PTpa_arg()
        PTpa_arg_init(pa_arg)
        
        pa_arg.infoSend.annId.annType = ESRF_INFO_ANN_ID
        pa_arg.infoSend.annId.var.annId = 0x84080006

        pa_arg.infoSend.numOfRepeat = 1
        pa_arg.infoSend.duration    = CC_TIMER_WAIT_ANS
        
        self.SendPA(CC_CALLER_LEG, pa_arg)         
        return
     
    
    def playInvalidCard(self):
        self.Trace("    playInvalidCard")
        
        pa_arg = PTpa_arg()
        PTpa_arg_init(pa_arg)
        
        pa_arg.infoSend.annId.annType = ESRF_INFO_ANN_ID
        pa_arg.infoSend.annId.var.annId = 0x84080007        

        pa_arg.infoSend.numOfRepeat = 1
        pa_arg.infoSend.duration    = CC_TIMER_WAIT_ANS
        
        self.SendPA(CC_CALLER_LEG, pa_arg) 
        return   
    
    
    def playBye(self):
        self.Trace("    playBye")
        
        pa_arg = PTpa_arg()
        PTpa_arg_init(pa_arg)
        
        pa_arg.infoSend.annId.annType = ESRF_INFO_ANN_ID
        pa_arg.infoSend.annId.var.annId = 0x84080008

        pa_arg.infoSend.numOfRepeat = 1
        pa_arg.infoSend.duration    = CC_TIMER_WAIT_ANS
        
        self.SendPA(CC_CALLER_LEG, pa_arg) 
        return
            
        
    def playWaitMusic(self):
        self.Trace("    playWaitMusic")
        
        self.SendRingBackToneToCaller()       
        return


    def connectDestNbr(self):
        self.Trace("connectDestNbr")
        
        conn_arg = PTconnect_arg()
        PTconnect_arg_init(conn_arg)
        
        conn_arg.calledNbr = self.destNbr
        
        conn_arg.dialPlanInd = MSS_PY_YES
        conn_arg.dialPlan = self.dialPlan
        
        conn_arg.servInfoInd = MSS_PY_YES
        conn_arg.servInfo.suppressCallLevel = SERVINFO_SUPPRESS_CALL_LEVEL_YES
        
        self.SendConnectWithArg(conn_arg) 
        return       
        
    
    def calculateCdrFee(self):
        self.Trace("calculateCdrFee")

        if 0 == self.answerTime:
            self.cdrFee = 0
            self.cdrDuration = 0
            return
        
        if self.releaseTime > self.answerTime :
            duration = self.releaseTime - self.answerTime
            self.cdrDuration = duration.seconds
        
        if 0 == self.rateInfo.tariff: 
            self.cdrFee = 0
        
        elif self.cdrDuration >= self.allowDuration:
            self.cdrFee = self.balance
        
        else:
            self.cdrFee = (self.cdrDuration / self.rateInfo.interval)*self.rateInfo.tariff            
            
            # even it is less than an interval, treat it as an interval         
            if 0 < (self.cdrDuration%self.rateInfo.interval):
                self.cdrFee += self.rateInfo.tariff
            
            # add one-time fee
            if 0 < self.cdrDuration:
                self.cdrFee += self.rateInfo.onetimeFee
                
            if self.cdrFee > self.balance:
                self.cdrFee = self.balance
        
        self.Trace("    CDR fee is %d" % self.cdrFee)   
             
        return
    
    def calculateDuration(self):
        self.Trace("calculateDuration")
        
        self.rateId = self.dbFeeMatrixMatch(self.dialPlan, self.destNbr)
        if 0 > self.rateId:
            self.Trace("    cannot get rate id.")            
            return -1
        
        result = self.dbGetRateInfo()
        if 0 > result:
            self.Trace("    cannot get rate info.")            
            return -2
        
        if 0 == self.rateInfo.tariff:
            self.allowDuration = CC_MAX_DURATION_FOR_FREE_CALL
        
        else:
            balance = self.balance - self.rateInfo.onetimeFee
            if balance < self.rateInfo.tariff:
                self.Trace("    The balance is not enough.")                
                return -3
            
            self.allowDuration = (balance / self.rateInfo.tariff)*self.rateInfo.interval
        
        return 1
    
    
    def updateBalance(self):
        if 0 == self.cdrFee:
            return
        
        balance = self.dbGetBalance()
        
        if balance < self.cdrFee:
            balance = 0
        
        else:
            balance -= self.cdrFee    
            
        self.dbSetBalance(balance)
        return
            
    #========================
    #
    # database functions
    #
    #========================
    def dbHasDialPlanInd(self, option):
        result = option & (1<<0)
        return result
    
    
    def dbGetCardInfo(self, card):
        
        sqlStr = "select cPassword,nBalance,cDialPlan, nOption from tbl_callingcard_balance " \
                 "where cCard='%s'" % card
        
        dba = Cdba_query()
        result = dba.query(sqlStr)
        if MSS_ERR_OK != result or 0 == dba.getRecordCnt():
            return -1

        dba.fetchRow()
        
        rowidx = 0
        
        self.pwd = dba.getRowVal(rowidx)
        rowidx+=1
        
        self.balance = dba.getRowIntVal(rowidx)
        rowidx+=1
        
        dialPlan = dba.getRowVal(rowidx)
        rowidx+=1
        
        option = dba.getRowIntVal(rowidx)
        rowidx+=1
        
        if 0 < self.dbHasDialPlanInd(option):
            self.dialPlan = dialPlan
            self.dialPlanInd = MSS_PY_YES
            
        return 0
    
    def dbGetBalance(self):
        if 0 == len(self.card):
            return 0
        
        sqlStr = "select nBalance from tbl_callingcard_balance where cCard='%s';" \
                 % self.card
        
        dba = Cdba_query()
        result = dba.query(sqlStr)
        if MSS_ERR_OK != result or 0 == dba.getRecordCnt():
            return -1
        
        dba.fetchRow()
        return dba.getRowIntVal(0)
    
    def dbSetBalance(self, new_balance):
        if 0 == len(self.card):
            return
        
        sqlStr = "update tbl_callingcard_balance set nBalance='%d' where cCard='%s';" \
                 % (new_balance, self.card)
        
        dba = Cdba_query()
        dba.query(sqlStr)
        return
    

    def dbFeeMatrixGet(self, dial_plan, called_nbr):
        sqlStr = "select nRateId from tbl_callingcard_fee_matrix " \
                 "where cDialPlan = '%s' and cRouteNbr='%s';" % \
                 (dial_plan, called_nbr)
        
        dba = Cdba_query()        
        result = dba.query(sqlStr)
        if MSS_ERR_OK != result:
            return -1 
        
        if 0 >= dba.getRecordCnt():
            return -2
        
        dba.fetchRow()        
        return dba.getRowIntVal(0)        
    
    
    def dbFeeMatrixMatchBase(self, dial_plan, called_nbr):
        sqlStr = "select nRateId from tbl_callingcard_fee_matrix where " \
                 "cDialPlan='%s' and position(cRouteNbr in '%s')=1 order by length(cRouteNbr) desc;" \
                 % (dial_plan, called_nbr)
        
        dba = Cdba_query()        
        result = dba.query(sqlStr)
        if MSS_ERR_OK != result:
            return -1 
        
        if 0 >= dba.getRecordCnt():
            return -2
        
        dba.fetchRow()        
        return dba.getRowIntVal(0)
   
    def dbFeeMatrixMatch(self, dial_plan, called_nbr):
       
        result = self.dbFeeMatrixMatchBase(dial_plan, called_nbr)
        if 0 < result:
            return result
        
        result = self.dbFeeMatrixGet(dial_plan, CC_DEFAULT_MATCH_TAG)
        if 0 < result:
            return result
        
        if CC_DEFAULT_DIAL_PLAN == dial_plan:
            return -1
        
        return self.dbFeeMatrixMatch(CC_DEFAULT_DIAL_PLAN, called_nbr)
       
       
    def dbGetRateInfo(self):
        if 0 >= self.rateId:
            return -1

        sqlStr = "select tariff_interval, tariff, onetime_fee from tbl_prepaid_rate " \
                 "where rate_id=%d;" % self.rateId
        
        dba = Cdba_query()
        result = dba.query(sqlStr)
        if MSS_ERR_OK != result or 0 == dba.getRecordCnt():
            return -2

        dba.fetchRow()        
        self.rateInfo.interval = dba.getRowIntVal(0)
        self.rateInfo.tariff   = dba.getRowIntVal(1)
        self.rateInfo.onetimeFee = dba.getRowIntVal(2)
        return 1
    
    
    def dbSetCDR(self):
        self.Trace("\t dbSetCDR")
        
        if 0 == len(self.card) or 0 == self.answerTime or MSS_PY_YES == self.setCDRInd:
            return
        
        sqlStr = "insert into tbl_callingcard_cdr (cCard, cCalled, StartTime, ConnTime, ReleaseTime, nDuration, nFee) " \
                 "values('%s', '%s', '%s', '%s', '%s', %d, %d);" % \
                 (self.card, self.destNbr, self.setupTime, self.answerTime, self.releaseTime, self.cdrDuration, self.cdrFee)
        
        dba = Cdba_query()
        dba.query(sqlStr)        
                
        self.initCDRInfo()        
        self.setCDRInd = MSS_PY_YES        
        return
        
