"""
=======================================================================
miniSIPServer hunting-group service script

Copyright (C) MYVOIPAPP,Inc. All Rights Reserved.

Author: Gilson

History:
2014-02-24 Migrated from MSS core.
=======================================================================
"""

from mss_datatype import *
from mss_service_basic import *

# HG strategy (defined in core)
ELH_NULL = 0
ELH_ROUND_ROBIN = 1
ELH_LINEAR      = 2

# operator status
EHG_OPER_CALL_IDLE = 0
EHG_OPER_CALL_BUSY = 1

# HG state
EHG_IDLE = 0
EHG_WAIT_OP_ANSWER          = 1
EHG_WAIT_PROMPT_OP_BUSY_END = 2
EHG_WAIT_OP_IDLE            = 3

#HG audio ID
HG_ANN_ID_MUSIC     = 0x00080002
HG_ANN_ID_OPER_BUSY = 0x08080001

# noinspection PyUnusedLocal
class MSS_Service(MSS_Basic_Service):

    def __init__(self):
        MSS_Basic_Service.__init__(self)

        # HG information
        self.groupID = 0
        self.hgStrategy = ELH_NULL
        self.hgNoAnswer = 0
        self.hgCallQueueInd = MSS_PY_NO
        self.hgMaxWaitCnt = 0
        self.hgMaxWaitDuration = 60

        # caller SRF information
        self.callerSrfID = 0

        # operator SRF information
        self.opSrfID = 0

        # run-time variants
        self.operator = ""
        self.addToQueueInd = MSS_PY_NO
        self.currRoundRobin = 0
        self.ctrInd = MSS_PY_NO
        return


    def DestroyProc(self):  #Destroy process
        self.Trace("DestroyProc")
        self.removeFromCallQueue()
        return


    def MainProc(self, currMsg): #main function
        try:
            if self.state == EHG_IDLE:
                self.onIdleProc()

            elif self.state == EHG_WAIT_OP_ANSWER:
                self.onWaitOperatorAnswer(currMsg)

            elif self.state == EHG_WAIT_PROMPT_OP_BUSY_END:
                self.onWaitAnnEnd(currMsg)

            elif self.state == EHG_WAIT_OP_IDLE:
                self.onWaitOPIdle(currMsg)

            else:
                self.Trace("unknown state(%d)." % self.state)
                self.EndService()

        except:
            self.HandleExcept()
        return


    ####################################################################
    #
    # State functions
    #
    ####################################################################
    def onIdleProc(self):
        self.Trace("onIdleProc: hunting-group")
        result = self.saveIDPInfo()
        if result < 0:
            self.SendReleaseCall()
            self.EndService()
            return

        result = self.dbGetHGInfo()
        if result < 0:
            self.SendReleaseCall()
            self.EndService()
            return

        self.monitorFirstCall()

        self.callNextOperator()
        return


    def onWaitOperatorAnswer(self, currMsg):
        self.Trace("onWaitOperatorAnswer")
        if currMsg == ESCP_EVT_ICA_RESP:
            self.onWaitOperatorAnswer_recvIcaResp()

        elif currMsg == ESCP_EVT_ERB:
            self.onWaitOperatorAnswer_ERB()

        elif currMsg == ESCP_EVT_TRANS_CLOSE:
            self.onWaitOperatorAnswer_TransClose()

        elif currMsg == ESCP_EVT_TIMEOUT:
            self.onWaitOperatorAnswer_TimeOut()

        else:
            self.Trace("\t strange. unknown message: %d" % currMsg)
            pass
        return


    def onWaitOperatorAnswer_recvIcaResp(self):
        self.Trace("onWaitOperatorAnswer_recvIcaResp")
        self.monitorSecondCall()
        self.SendContinue(1)
        return


    def onWaitOperatorAnswer_ERB(self):
        self.Trace("onWaitOperatorAnswer_ERB")
        if self.edpEvent == DP_O_ANSWER:
            self.opSrfID = self.currMessage['calledSrfID']
            self.operatorAnswer()

        else:
            self.Trace("\t first call is released, destroy whole service.")
            self.exceptionProc()

        return


    def onWaitOperatorAnswer_TransClose(self):
        self.Trace("onWaitOperatorAnswer_TransClose")
        if self.GetSsfFsmID(0) == MSS_INVALID_OBJ_ID:
            # if the message is from first call, that means exception
            self.exceptionProc()
        else:
            self.operatorIdle()
            self.callNextOperator()

        return


    def onWaitOperatorAnswer_TimeOut(self):
        self.Trace("onWaitOperatorAnswer_TimeOut")
        self.SendReleaseCall(CAUSE_NORMAL_UNSPECIFIED, 1)
        self.operatorIdle()
        self.callNextOperator()
        return


    def callNextOperator(self):
        self.Trace("\t callNextOperator")
        self.StopTimer()
        self.operator = ""

        result = self.findOperator()
        if result != MSS_ERR_OK:
            if result == MSS_ERR_BUSY:
                self.allOperatorsBusy()

            else:
                self.SendReleaseCall()
                self.EndService()
            return

        self.playMusic()
        self.callOperator()

        self.startWaitOperatorTimer()
        self.EnterState(EHG_WAIT_OP_ANSWER)
        return


    def onWaitAnnEnd(self, currMsg):
        self.Trace("onWaitAnnEnd")
        if currMsg == ESCP_EVT_SRR:
            self.onWaitAnnEnd_SRR()

        elif currMsg == ESCP_EVT_DB_OPERATOR_IDLE:
            self.SaveMessage()

        else:
            self.StopTimer()
            self.exceptionProc()
        return


    def onWaitAnnEnd_SRR(self):
        self.Trace("onWaitAnnEnd_SRR")
        self.StopTimer()

        self.playMusic()
        self.StartTimer(self.hgMaxWaitDuration)
        self.EnterState(EHG_WAIT_OP_IDLE)
        return


    def onWaitOPIdle(self, currMsg):
        self.Trace("onWaitOPIdle")
        if currMsg == ESCP_EVT_DB_OPERATOR_IDLE:
            self.onWaitOPIdle_OperatorIdle()

        else:
            self.StopTimer()
            self.exceptionProc()
        return


    def onWaitOPIdle_OperatorIdle(self):
        self.Trace("onWaitOPIdle_OperatorIdle")

        self.StopTimer()
        self.removeFromCallQueue()

        self.callNextOperator()
        return


    ####################################################################
    #
    # Event functions
    #
    ####################################################################
    def saveIDPInfo(self):
        self.Trace("\t saveIDPInfo")
        self.callerSrfID = self.currMessage['callerSrfID']

        result = self.getGroupID()
        if result < 0:
            self.Trace("\t fail to detect group ID. errCode=%d" % result)
            return -1
        return 0


    def exceptionProc(self):
        self.Trace("\t exceptionProc")
        self.releaseAllSSF()
        self.operatorIdle()
        self.EndService()
        return


    def monitorFirstCall(self):
        self.Trace("\t monitorFirstCall")
        self.SendRRBE(PASSIVE_LEGID, DP_T_DISCONNECT,DP_MON_NOTIFY)
        return

    def monitorSecondCall(self):
        self.Trace("\t monitorSecondCall")
        self.SendRRBE(PASSIVE_LEGID, DP_O_ANSWER,DP_MON_INTERUPT,1)
        return

    def findOperator(self):
        self.Trace("\t findOperator")

        result = self.getOperator()
        if result == MSS_ERR_OK:
            return MSS_ERR_OK

        elif result == MSS_ERR_BUSY:
            self.Trace("\t\t cannot find an idle operator")
            return MSS_ERR_BUSY

        else:
            self.Trace("\t\t no operation.")
            return MSS_ERR_OBJ_NOT_FOUND


    def playMusic(self):
        self.Trace("\t playMusic")
        self.sendCTR()

        pa_arg = PTpa_arg()
        PTpa_arg_init(pa_arg)

        pa_arg.infoSend.annId.annType = ESRF_INFO_ANN_ID
        pa_arg.infoSend.annId.var.annId = HG_ANN_ID_MUSIC
        pa_arg.infoSend.duration = 3600

        self.SendPA(PASSIVE_LEGID, pa_arg)
        return


    def discMusic(self):    # disconnect music
        self.Trace("\t discMusic")
        if self.ctrInd == MSS_PY_NO:
            return

        self.SendDFC(PASSIVE_LEGID)
        self.ctrInd = MSS_PY_NO
        return


    def callOperator(self):
        self.Trace("\t callOperator")

        ica_arg = PTica_arg()
        PTica_arg_init(ica_arg)

        ica_arg.calledNbr = self.operator

        ica_arg.callerNbrInd = MSS_PY_YES
        ica_arg.callerNbr = self.callerNbr

        ica_arg.callerSrfInfoInd = MSS_PY_YES
        ica_arg.callerSrfID = self.callerSrfID

        self.SendICA(ica_arg,1)
        return


    def startWaitOperatorTimer(self):
        self.Trace("\t startWaitOperatorTimer")
        self.StartTimer(self.hgNoAnswer)
        return


    def allOperatorsBusy(self):
        self.Trace("\t allOperatorsBusy")
        if self.hgCallQueueInd == MSS_PY_NO:
            self.Trace("\t\t this group can't support queue, so release call.")
            self.SendReleaseCall()
            self.EndService()
            return MSS_ERR_INVALID_OPR

        queueID = self.cdbGetCallQueueCnt(self.groupID)
        if queueID < 0 or queueID > self.hgMaxWaitCnt:
            self.Trace("\t\t cannot add call to queue.")
            self.SendReleaseCall()
            self.EndService()
            return MSS_ERR_FAIL

        result = self.cdbAddToCallQueue(self.groupID)
        if result != MSS_ERR_OK:
            self.Trace("\t\t fail to add into queue.")
            self.SendReleaseCall()
            self.EndService()
            return MSS_ERR_OUT_OF_MEMORY

        queueID += 1
        self.addToQueueInd = MSS_PY_YES

        self.promptAllOperatorsBusy(queueID)

        self.StartTimer(self.hgMaxWaitDuration)
        self.EnterState(EHG_WAIT_PROMPT_OP_BUSY_END)
        return MSS_ERR_OK


    def operatorAnswer(self):
        self.Trace("\t operatorAnswer")

        self.StopTimer()
        self.discMusic()
        self.bridge2Parties()
        self.SendContinue(1)
        self.endAllSsf()
        self.EndService()
        return

    ###################################################################
    #
    # common functions
    #
    ####################################################################
    def getGroupID(self):
        self.Trace("\t getGroupID")

        dialPlan = self.origDialPlan
        result = self.dbGetHGGroupID(dialPlan)
        if result < 0:
            if dialPlan != MSS_DEFAULT_DIAL_PLAN:
                result = self.dbGetHGGroupID(MSS_DEFAULT_DIAL_PLAN)
                if result < 0:
                    return -2
            else:
                return -1

        self.groupID = result
        self.Trace("\t\t group ID is %d" % result)
        return 0


    def removeFromCallQueue(self):
        self.Trace("\t removeFromCallQueue")
        if self.addToQueueInd == MSS_PY_YES:
            self.dbRemoveFromCallQueue(self.groupID, self.scpFsmId)
            self.addToQueueInd = MSS_PY_NO
        return


    def releaseAllSSF(self):
        self.Trace("\t releaseAllSSF")
        self.SendReleaseCall(CAUSE_NORMAL_UNSPECIFIED, 0)
        self.SendReleaseCall(CAUSE_NORMAL_UNSPECIFIED, 1)
        return


    def getOperator(self):
        self.Trace("\t getOperator")

        loginCnt = self.cdbHGOperatorGetLoginCnt()
        self.Trace("\t\t login count is %d" % loginCnt)
        if loginCnt <= 0:
            return MSS_ERR_OBJ_NOT_FOUND

        idleCnt = self.cdbHGOperatorGetIdleCnt()
        self.Trace("\t\t idle operators count is %d" % idleCnt)
        if idleCnt <= 0:
            return MSS_ERR_BUSY

        if self.hgStrategy == ELH_ROUND_ROBIN:
            result = self.getOperatorByRoundRobin()
        elif self.hgStrategy == ELH_LINEAR:
            result = self.getOperatorByLinear()
        else:
            self.Trace("\t\t unknown strategy: %d" % self.hgStrategy)
            return MSS_ERR_INVALID_OPR

        return MSS_ERR_OK


    def getOperatorByRoundRobin(self):
        self.Trace("\t getOperatorByRoundRobin")
        result = self.cdbHGOperatorGetOperatorForRoundRobin()
        if result != MSS_ERR_OK:
            return MSS_ERR_OBJ_NOT_FOUND

        self.cdbHGOperatorSetRoundRobinCnt(self.operator, self.currRoundRobin)
        self.cdbHGOperatorSetCallState(self.operator, EHG_OPER_CALL_BUSY)
        return MSS_ERR_OK


    def getOperatorByLinear(self):
        self.Trace("\t getOperatorByLinear")
        result = self.cdbHGOperatorGetOperatorForLinear()
        if result != MSS_ERR_OK:
            return MSS_ERR_OBJ_NOT_FOUND

        self.cdbHGOperatorSetCallState(self.operator, EHG_OPER_CALL_BUSY)
        return MSS_ERR_OK

    def sendCTR(self):
        if self.ctrInd == MSS_PY_YES:
            return
        self.Trace("\t sendCTR")

        self.ctrInd = MSS_PY_YES
        self.SendCTR(PASSIVE_LEGID)
        return

    def promptAllOperatorsBusy(self, queueID):
        self.Trace("\t promptAllOperatorsBusy")
        self.sendCTR()

        pa_arg = PTpa_arg()
        PTpa_arg_init(pa_arg)

        pa_arg.infoSend.annId.annType = ESRF_INFO_ANN_ID
        pa_arg.infoSend.annId.var.annId = HG_ANN_ID_OPER_BUSY
        pa_arg.infoSend.numOfRepeat = 1
        pa_arg.infoSend.infoVar.varType = ESRF_INFO_VAR_DIGIT
        pa_arg.infoSend.infoVar.var.digit = queueID

        self.SendPA(PASSIVE_LEGID, pa_arg)
        return


    def bridge2Parties(self):
        self.Trace("\t bridge2Parties")
        b2p_arg = PTbridge2parties_arg()
        PTbridge2parties_arg_init(b2p_arg)

        b2p_arg.peerSrfID = self.opSrfID
        b2p_arg.peerSsfFsmID = self.GetSsfFsmID(1)
        self.SendBridge2Parties(b2p_arg)
        return


    def endAllSsf(self):
        self.Trace("\t endAllSsf")
        self.SendTransClose(0)
        self.SendTransClose(1)
        return

    ###################################################################
    #
    # local database functions
    #
    ####################################################################
    def dbGetHGInfo(self):
        self.Trace("\t dbGetHGInfo")

        sqlStr = "select strategy, noAnswer, queueInd, maxWaitCnt, maxWaitDuration " \
                 " from tbl_hunting_group where groupID=%u" % self.groupID
        result = self.ldbQuery(sqlStr)
        if result != MSS_ERR_OBJ_FOUND:
            return -1

        self.hgStrategy = self.ldbGetRowIntVal(0)
        self.hgNoAnswer = self.ldbGetRowIntVal(1)
        self.hgCallQueueInd = self.ldbGetRowIntVal(2)
        self.hgMaxWaitCnt = self.ldbGetRowIntVal(3)
        self.hgMaxWaitDuration = self.ldbGetRowIntVal(4)
        return 0

    ###################################################################
    #
    # cache database functions
    #
    ####################################################################
    def dbGetHGGroupID(self, dialPlan):
        sqlStr = "select groupID from tbl_hunting_group_detect" \
                " where dialPlan='%s' and calledNbr='%s'" \
                % (dialPlan, self.calledNbr)

        result = self.cdbQuery(sqlStr)
        if result != MSS_ERR_OBJ_FOUND:
            return -1

        groupID = self.cdbGetRowIntVal(0)
        return groupID


    def dbRemoveFromCallQueue(self, groupID, scpFsmID):
        if groupID == MSS_INVALID_OBJ_ID or scpFsmID == MSS_INVALID_OBJ_ID:
            return -1

        sqlStr = "delete from cache_hg_queue where " \
                 " groupID=%d and fsmID=%d" % (groupID, scpFsmID)

        result = self.cdbQuery(sqlStr)
        return 0


    def cdbHGOperatorGetLoginCnt(self):
        sqlStr = "select count(*) from cache_hg_operator where groupID=%d" % self.groupID

        result = self.cdbQuery(sqlStr)
        if result != MSS_ERR_OBJ_FOUND:
            return -1

        result = self.cdbGetRowIntVal(0)
        return result


    def cdbHGOperatorGetIdleCnt(self):
        sqlStr = "select count(*) from cache_hg_operator where " \
                 "groupID=%d and callState=%d" % (self.groupID, EHG_OPER_CALL_IDLE)

        result = self.cdbQuery(sqlStr)
        if result != MSS_ERR_OBJ_FOUND:
            return -1

        result = self.cdbGetRowIntVal(0)
        return result


    def cdbHGOperatorSetCallState(self, operator, callState):
        sqlStr = "update cache_hg_operator set callState=%u where operator='%s'" \
                    % (callState, operator)

        result = self.cdbQuery(sqlStr)
        if result != MSS_ERR_OK:
            self.Trace("\t\t SQL error: %s" % sqlStr)
        return


    def cdbHGOperatorSetRoundRobinCnt(self, operator, cnt):
        sqlStr = "update cache_hg_operator set roundRobinCnt=%d where operator='%s'" \
                    % (cnt, operator)
        self.cdbQuery(sqlStr)
        return


    def cdbHGOperatorGetOperatorForRoundRobin(self):
        sqlStr = "select operator, roundRobinCnt from cache_hg_operator where "\
                "groupID=%u and callState=%u order by roundRobinCnt limit 1" \
                 % (self.groupID, EHG_OPER_CALL_IDLE)
        result = self.cdbQuery(sqlStr)
        if result != MSS_ERR_OBJ_FOUND:
            return MSS_ERR_OBJ_NOT_FOUND

        self.operator = self.cdbGetRowVal(0)
        self.currRoundRobin = self.cdbGetRowIntVal(1)
        self.currRoundRobin += 1

        self.Trace("\t\t operator='%s', roundRobin=%d" %(self.operator, self.currRoundRobin))
        return MSS_ERR_OK


    def cdbHGOperatorGetOperatorForLinear(self):
        sqlStr = "select operator from cache_hg_operator where groupID=%u and callState=%u "    \
                 "order by loginTime limit 1" %(self.groupID, EHG_OPER_CALL_IDLE)
        result = self.cdbQuery(sqlStr)
        if result != MSS_ERR_OBJ_FOUND:
            return MSS_ERR_OBJ_NOT_FOUND

        self.operator = self.cdbGetRowVal(0)
        return MSS_ERR_OK


    def cdbGetCallQueueCnt(self, groupID):
        sqlStr = "select count(*) from cache_hg_queue where groupID=%u" % groupID
        result = self.cdbQuery(sqlStr)
        if result != MSS_ERR_OBJ_FOUND:
            return -1

        result = self.cdbGetRowIntVal(0)
        return result


    def cdbAddToCallQueue(self, groupID):
        currSeconds = self.getCurrSeconds()
        sqlStr = "insert into cache_hg_queue (groupID, fsmID, loginTime) "  \
                 "values(%d, %d, %d)" % (groupID, self.scpFsmId, currSeconds)

        result = self.cdbQuery(sqlStr)
        if result != MSS_ERR_OK:
            self.Trace("\t\t SQL: %s" % sqlStr)
            return MSS_ERR_FAIL
        else:
            return MSS_ERR_OK

    ###################################################################
    #
    # Core API functions
    #
    ###################################################################
    def operatorIdle(self):
        self.Trace("\t\t operatorIdle")
        result = self.ifDll.PyHGOperatorIdle(self.operator)
        return result