#==============================================================================
# miniSipServer Caller prepaid service script
#
# Copyright (C) 2007-2011, MyVoipApp,Inc. All Rights Reserved.
# 
# author: Gilson
#
# History:
# 2011-03-05 Migrated from MSS core.      
# 2011-08-30 Support one-time fee in prepaid rate
#==============================================================================

from datetime import *

from mss_datatype import *
from mss_dba import *
from mss_service_basic import *


#service states
ECALLER_PREPAID_IDLE            = 0
ECALLER_PREPAID_WAIT_ANS        = 1
ECALLER_PREPAID_WAIT_DISC       = 2
ECALLER_PREPAID_WAIT_ANN_END    = 3

#default value
DEFAULT_MATCH_TAG           = "*"

TIMER_WAIT_ANS = 120 # 120 seconds

class Prate_info():
    def __init__(self):
        self.interval = 0
        self.tariff   = 0
        self.onetimeFee = 0

class MSS_Service(MSS_Basic_Service):

    def MainProc(self, currMsg): #main function
        self.Trace("CallerPrepaid:MainProc")
        
        try:
            if self.state == ECALLER_PREPAID_IDLE:
                self.OnIdleProc(currMsg)
            
            elif self.state ==  ECALLER_PREPAID_WAIT_ANS:
                self.OnWaitAnsProc(currMsg)
            
            elif self.state == ECALLER_PREPAID_WAIT_DISC:
                self.OnWaitDiscProc(currMsg)
            
            elif self.state == ECALLER_PREPAID_WAIT_ANN_END:
                self.OnWaitAnnEndProc(currMsg)
                
            else:
                self.Trace("unknown state %d" % self.state)
                self.EndService()
                
        except:
            self.HandleExcept()
            
        return
    
    def DestroyProc(self):  #Destroy process
        self.Trace("DestroyProc")
        
        self.releaseTime = datetime.now()
        
        if self.answerTime > self.setupTime:
            diff_time = self.releaseTime - self.answerTime
            self.duration = diff_time.seconds
        
        self.db_set_cdr()        
        return

    def OnIdleProc(self, currMsg):
        self.Trace("CallerPrepaid:OnIdleProc")
        
        # init self parameters        
        self.rateInfo = Prate_info()
        self.fee      = 0
        self.allowDuration = 0
        self.duration = 0
        self.expireInd = 0
        
        #record setup time
        self.setupTime      = datetime.now()
        self.answerTime     = self.setupTime
        self.releaseTime    = self.setupTime
        
        # get subscriber number
        if( 1 == self.foSSPInd):
            self.subNbr = self.redirNbr
        else:
            self.subNbr = self.callerNbr                        
        self.Trace("subscriber number is %s" % self.subNbr)
        
        # get subscriber rate id
        self.rateId = self.db_caller_prepaid_get_rateid()
        self.Trace("prepaid rate id is %d" % self.rateId)
        
        #get rate information
        result = self.db_caller_prepaid_get_rate_info()
        if 0 >= result:
            self.Trace("Fail to get rate information (%d)" % result)
            self._exception()
            return

        #get subscriber balance
        self.balance = self.db_caller_prepaid_get_sub_balance()
        self.Trace("the balance of current user is %d" % self.balance)

        #check balance
        result = self._checkBalance()
        if result<0: # no enough balance
            self.Trace("Fail to check balance (%d)" % result)
            
            if 1 == self.foSSPInd: # current SSP is a forwardin SSP, it is unnecessary to prompt audio
                self.Trace("without balance in fowarding call. Just release call.")
                self.SendReleaseCall(CAUSE_CALLER_BALANCE_NOT_ENOUGH)
                self.EndService()
                return
            else:
                self.SendCTR(CALLER_LEGID)
                self._sendPA_BalanceInsufficient()
                
                self.StartTimer(60)
                
                self.EnterState(ECALLER_PREPAID_WAIT_ANN_END)
                
                return
        elif result==0:
            self.SendContinue()
            self.EndService()
            return
                
        # the subscriber has enough balance
        self.SendRRBE(CALLED_LEGID, DP_O_ANSWER,        DP_MON_NOTIFY) 
        self.SendRRBE(CALLER_LEGID, DP_O_DISCONNECT,    DP_MON_NOTIFY)         
        self.SendContinue()
        
        self.StartTimer(TIMER_WAIT_ANS)    #wait answer
        self.EnterState(ECALLER_PREPAID_WAIT_ANS)
        
        return
    
    def OnWaitAnsProc(self, currMsg):
        self.Trace("OnWaitAnsProc")
        
        self.StopTimer()
        
        if currMsg == ESCP_EVT_ERB: # it must be ANSWER event
            self.RecvAnswer()
        
        elif currMsg == ESCP_EVT_TIMEOUT:
            self._exception()

        else:
            self.EndService()
            
        return
    
    def OnWaitDiscProc(self, currMsg):
        self.Trace("OnWaitDiscProc")
        
        self.StopTimer()
        
        if currMsg == ESCP_EVT_ERB:
            self.RelCall()
            
        elif currMsg == ESCP_EVT_TRANS_CLOSE:
            self.RelCall()
        
        elif currMsg == ESCP_EVT_TIMEOUT:
            self.BalanceExpired()
        
        else:
            self.EndService()
            
        return
    
    def OnWaitAnnEndProc(self, currMsg):
        self.Trace("OnWaitAnnEndProc")
        
        self.StopTimer()
        
        if currMsg == ESCP_EVT_TRANS_CLOSE :
            self.RelCall()
        
        elif currMsg == ESCP_EVT_TIMEOUT or \
             currMsg == ESCP_EVT_SRR or \
             currMsg == ESCP_EVT_SRF_ERR:
            self.RelCallForBalanceInsufficient()
        
        else:
            self.EndService()
    
        return
        
    def RecvAnswer(self):
        self.Trace("receive answer event")
        
        self.answerTime = datetime.now()
        
        self.StartTimer(self.allowDuration)
        self.EnterState(ECALLER_PREPAID_WAIT_DISC)
        
        return
    
    def RelCall(self):
        self.Trace("RelCall")
        self.releaseTime = datetime.now()
        
        self.calculateFee()
        
        self.EndService()
        
        return
    
    def RelCallForBalanceInsufficient(self):
        self.Trace("RelCallForBalanceInsufficient")
        
        self.StopTimer()
        
        self.SendReleaseCall(CAUSE_CALLER_BALANCE_NOT_ENOUGH)
        self.EndService()
        
        return
        
    def BalanceExpired(self):
        self.Trace("Balance is expired")
        
        self.expireInd = MSS_PY_YES
        
        self.releaseTime = datetime.now()
        
        self.SendReleaseCall()
        
        self.calculateFee()
        
        self.EndService()
        return
    ######################################
    #
    # Common functions
    #
    ######################################
    def _exception(self):
        self.Trace("_exception")

        self.SendReleaseCall()
        self.EndService()
        return

    def _checkBalance(self):
        if (0 == self.rateInfo.tariff and 0 == self.rateInfo.onetimeFee) \
        or (FREE_CALL_IND_YES == self.servInfoInd.freeCallInd):
            self.Trace("current call is a free call")
            return 0
        else:
            if 0 >= self.balance:
                self.Trace("Current user doesn't have enough balance")
                return -1
            
            balance = self.balance - self.rateInfo.onetimeFee;
            self.allowDuration = (balance/self.rateInfo.tariff) * self.rateInfo.interval
            self.Trace("balance=%d, tariff=%d, interval=%d, onetime_fee=%d, allow_duration=%d" % 
                       (self.balance, self.rateInfo.tariff, self.rateInfo.interval, self.rateInfo.onetimeFee,self.allowDuration))
            
        if self.allowDuration >= self.rateInfo.interval:
            return 1
        else:
            self.Trace("the balance cannot make a min-interval call.")
            return -2    
    
    def _sendPA_BalanceInsufficient(self):
        pa_arg = PTpa_arg()
        PTpa_arg_init(pa_arg)
        
        pa_arg.infoSend.annId.annType = ESRF_INFO_ANN_ID
        pa_arg.infoSend.annId.var.annId = 0x07080001
        
        pa_arg.infoSend.numOfRepeat = 1
        
        self.SendPA(CALLER_LEGID, pa_arg)
        
        return
    
    def calculateFee(self):
        if self.answerTime > self.setupTime:
            diff_time = self.releaseTime - self.answerTime        
            duration = diff_time.seconds
            
            # refine duration since mss timer has seceral seconds wrong.
            if MSS_PY_YES == self.expireInd:
                duration = self.allowDuration
        else:
            duration = 0
        
        if (0 == self.rateInfo.tariff) or (FREE_CALL_IND_YES == self.servInfoInd.freeCallInd):
            self.fee = 0
        
        else:
            fee = ( duration / self.rateInfo.interval ) * self.rateInfo.tariff
            
            # even it is less than a interval, treat it as an interval
            if 0 < duration % self.rateInfo.interval:
                fee += self.rateInfo.tariff
            
            # add one-time fee
            if 0 < duration:
                fee += self.rateInfo.onetimeFee
                
            self.fee = fee            
        
        # re-get balance from database since it is possible administrator update balance at the same time
        balance = self.db_caller_prepaid_get_sub_balance()
        
        self.Trace("balance=%d, fee=%d" % (balance, self.fee))
        
        balance -= self.fee
        
        self.db_caller_prepaid_set_sub_balance(balance)
        
        return
        
    ######################################
    #
    # database functions
    #
    ######################################    
        
    def _db_caller_prepaid_get_rateid(self, caller_nbr, called_nbr):        
        sqlStr = "select nRateId from tbl_caller_prepaid where cSubscriber='%s' " \
                 "and cRouteNbr ='%s'" % \
                 (caller_nbr, called_nbr)                 
        
        dba = Cdba_query()
        result = dba.query(sqlStr)
        if MSS_ERR_OK != result or 0 == dba.getRecordCnt():
            return 0
            
        dba.fetchRow()               
        return dba.getRowIntVal(0)

    def _db_caller_prepaid_get_rateid_for_any_sub(self,called_nbr):
        sqlStr = "select nRateId from tbl_caller_prepaid where " \
                 "cSubscriber='%s' and position(cRouteNbr in '%s')=1 order by length(cRouteNbr) desc limit 1" % \
                  (DEFAULT_MATCH_TAG, called_nbr)
        
        dba = Cdba_query()
        result = dba.query(sqlStr)
        if MSS_ERR_OK != result or 0 == dba.getRecordCnt():
            return 0
            
        dba.fetchRow()
        return dba.getRowIntVal(0)
            
    def db_caller_prepaid_get_rateid(self):
        rate_id = 0
        
        #search for special record
        sqlStr = "select nRateId from tbl_caller_prepaid where " \
                 "cSubscriber='%s' and position(cRouteNbr in '%s')=1 order by length(cRouteNbr) desc limit 1" % \
                 (self.subNbr, self.calledNbr)                 

        dba = Cdba_query()
        result = dba.query(sqlStr)
        if MSS_ERR_OK == result:
            dba.fetchRow()
            rate_id=dba.getRowIntVal(0)              
        
        if 0 == rate_id:
            #check any route number for the subscrier
            rate_id = self._db_caller_prepaid_get_rateid(self.subNbr, DEFAULT_MATCH_TAG)
        
        if 0 == rate_id:
            #check any subscriber for this route number
            rate_id = self._db_caller_prepaid_get_rateid_for_any_sub(self.calledNbr)
        
        if 0 == rate_id:
            #check any subscriber for any route number
            rate_id = self._db_caller_prepaid_get_rateid(DEFAULT_MATCH_TAG, DEFAULT_MATCH_TAG)
            
        return rate_id

    def db_caller_prepaid_get_rate_info(self):
        if 0 == self.rateId:
            return -1

        sqlStr = "select tariff_interval, tariff, onetime_fee from tbl_prepaid_rate where rate_id=%d" % \
                            self.rateId
        
        dba = Cdba_query()
        result = dba.query(sqlStr)
        if MSS_ERR_OK != result:
            return -2
            
        row_count = dba.getRecordCnt()
        if 1 != row_count:
            return -3

        dba.fetchRow()
        self.rateInfo.interval = dba.getRowIntVal(0)
        self.rateInfo.tariff   = dba.getRowIntVal(1)
        self.rateInfo.onetimeFee = dba.getRowIntVal(2)        
        return 1

    def db_caller_prepaid_get_sub_balance(self):
        
        sqlStr = "select nBalance from tbl_sub_balance where cSubName='%s'" % \
                  self.subNbr
        
        dba = Cdba_query()
        result = dba.query(sqlStr)
        if MSS_ERR_OK != result:
            return 0
            
        if 0 == dba.getRecordCnt():
            return 0

        dba.fetchRow()
        return dba.getRowIntVal(0)
    
    def db_caller_prepaid_set_sub_balance(self, balance):
        sqlStr = "update tbl_sub_balance set nBalance=%d where cSubName='%s'" % \
                   (balance, self.subNbr)
                
        dba = Cdba_query()
        dba.query(sqlStr)
        return
    
    def db_set_cdr(self):
        if 0 == self.duration:
            return
                                
        sqlStr = "insert into tbl_cdr " \
                 "(nDirection, cCaller, cCalled, nSubType, tStartTime, tConnectTime,tReleaseTime,nDuration,nFee) "   \
                 "values (0,'%s','%s',1,'%s','%s','%s',%d,%d)" % \
                 (self.subNbr, self.calledNbr,self.setupTime, self.answerTime, self.releaseTime,self.duration, self.fee)
        
        dba = Cdba_query()
        dba.query(sqlStr)
        return
