#==============================================================================
# miniSipServer basic service script (All services should be its sub-class)
#
# Copyright (C) 2007-2011, MyVoipApp,Inc. All Rights Reserved.
# 
# All contents are system definition, customers SHOULD NOT modify them!
# 
# author: Hong
#==============================================================================

#DP monitor type
DP_MON_INTERUPT = 0
DP_MON_NOTIFY   = 1

#DPs
DP_ORIG_ATTEMPT_AUTHORIZED = 1
DP_COLLECTED_INFO          = 2
DP_ANALYZED_INFO           = 3
DP_ROUTE_SELECT_FAILURE    = 4
DP_O_BUSY                  = 5
DP_O_NO_ANSWER             = 6
DP_O_ANSWER                = 7
DP_O_MIDCALL               = 8
DP_O_DISCONNECT            = 9
DP_O_ABANDON               = 10
DP_TERM_ATTEMPT_AUTHORIZED = 12
DP_T_BUSY                  = 13
DP_T_NO_ANSWER             = 14
DP_T_ANSWER                = 15
DP_T_MID_CALL              = 16
DP_T_DISCONNECT            = 17
DP_T_ABANDON               = 18
DP_O_TERM_SEIZED           = 19
DP_ORIG_ATTEMPT            = 22
DP_TERM_ATTEMPT            = 23
DP_CALL_ACCEPTED           = 27

#Messages
ESCP_EVT_UNKNOWN            = 0
ESCP_EVT_TIMEOUT            = 1
ESCP_EVT_IDP                = 2
ESCP_EVT_ERB                = 3
ESCP_EVT_TRANS_CLOSE        = 4
ESCP_EVT_ICA_RESP           = 5
ESCP_EVT_CALLBACK_REQ       = 6
ESCP_EVT_SRR                = 7
ESCP_EVT_PNC_RESULT         = 8
ESCP_EVT_SRF_ERR            = 9
ESCP_EVT_SRF_RECORD_ANN_END = 10
ESCP_EVT_ERROR              = 11
ESCP_EVT_LEG_RELD           = 12
ESCP_EVT_CALLBACK_REL       = 13
ESCP_EVT_SRF_ANN_END        = 50
ESCP_EVT_SRF_BIND_RESP      = 51
ESCP_EVT_SCF_CALL_RETRIEVE_REQ  = 70
ESCP_EVT_SCF_CALL_RETRIEVE_RESP = 71
ESCP_EVT_DB_CONF_MEMBER_ATTEND = 90
ESCP_EVT_DB_OPERATOR_IDLE      = 91
ESCP_EVT_UA_USER_IDLE = 100

#LegId (O side)
CALLER_LEGID  = 1
CALLED_LEGID  = 2

#legID
CONTROL_LEGID = 1
PASSIVE_LEGID = 2

#process result.
PRC_PROCESSED  = 1
PRC_DISCARD    = 2
PRC_KILL       = 3
PRC_SAVE       = 10 

# Call stage, same with Ecall_stage
ECALL_DIAL  = 0
ECALL_ALERT = 1
ECALL_TALK  = 2

# SRF error type, same with Esrf_err_type defined in mss core
ESRF_ERR_OK = 0
ESRF_ERR_INVALID_OBJ = 1
ESRF_ERR_WRONG_SEQUENCE = 2
ESRF_ERR_UNAVAILABLE_RES = 3
ESRF_ERR_FORCE_RELEASED = 4
ESRF_ERR_FIRST_DIGIT_TIMEOUT = 5
ESRF_ERR_IMPROPER_CALLER_RESPONSE = 6
ESRF_ERR_UNKNOWN = 255

#common announcement id 
ANN_ID_COMMON_MUSIC     = 0x00080002
ANN_ID_COMMON_RINGBACK  = 0x00080003
ANN_ID_OP_SUCCESS       = 0X01080001

import os as _os
import sys
import string
import traceback
from ctypes import *

from mss_datatype import *                   

class MSS_Basic_Service:
    def __init__(self):
        
        # initiate service data which will be used by all sub-class
        self.state     = 0
        self.callerNbr = ""
        self.calledNbr = ""
        self.redirNbr  = ""
        self.origDialPlan = "" # Original dial plan
        self.ivrXMLServiceID = 0    # inner service file ID

        #service information indication
        self.servInfoInd = PTserv_info_ind()

        #PNC result
        self.pncResult  = ""
        self.srfError   = ESRF_ERR_OK
        
        #Current EDP information
        self.edpEvent  = 0
        self.edpLegId  = 0

        #message detail from SSF or SRF
        self.currMessage = {}

        # system parameters (CANNOT be modified)
        self.scpFsmId   = 0
        self.profileID  = 0
        self.foSSPInd   = 0      # Forwarding SSP indication
        self.tracedInd  = MSS_PY_NO

        self.prcResult  = PRC_PROCESSED
        
        if _os.name == "posix":
            self.ifDll = CDLL("libmsscore.so")
        else:
            self.ifDll = CDLL("msscore.dll")
        
        return   
    
    def getExeDirName(self):
        exeDir = _os.path.dirname(_os.path.realpath(__file__))+"/.."
        return exeDir
        
    #=================================
    # log functions for scripts
    #================================= 
    def Trace(self, info):
        if self.tracedInd == MSS_PY_NO:
            return

        self.ifDll.PyTrace(self.scpFsmId, info)
        return
    
    def TraceError(self):
        exc_type, exc_value, exc_traceback = sys.exc_info()
        exc_list = traceback.format_exception(exc_type, exc_value,exc_traceback)        
        
        exc_len = len(exc_list)
        index = 0
       
        while exc_len > 0:
            self.Trace(exc_list[index])
            index += 1
            exc_len -= 1  
           
        return        
        
    #========================================
    # Public functions for script services
    #========================================
    def EnterState(self, new_state):
        self.Trace("Enter state : from %d to %d" %
                   (self.state, new_state))
        
        self.state = new_state
        return
    
    def HandleExcept(self):
        self.TraceError()
        self.SendReleaseCall()
        self.EndService()        
        return
        
    def StartTimer(self, seconds):
        self.Trace("\t StartTimer")
        
        self.ifDll.PyStartTimer(self.scpFsmId, seconds)
        return
        
    def StopTimer(self):
        self.Trace("\t StopTimer")
        self.ifDll.PyStopTimer(self.scpFsmId)
        return
    
    def EndService(self):        
        self.Trace("\t EndService")
        self.prcResult = PRC_KILL        
        return

    def SaveMessage(self):
        self.Trace("\t SaveMessage")
        self.prcResult = PRC_SAVE
        return

    def GetSsfFsmID(self, ssfIndex=0):
        result = self.ifDll.PyGetSsfFsmID(self.scpFsmId, ssfIndex)
        return result
    
    #========================================
    # Intelligent network operations
    #========================================
    def SendRRBE(self, leg_id, dp, monitor_type, ssfIndex=0): #RequestReportBCSMEvent
        self.Trace("\t\t SendRRBE")
        
        self.ifDll.SendRRBE( self.scpFsmId,leg_id, dp, monitor_type, ssfIndex)
        return
        
    def SendCTR(self, leg_id, ssfIndex=0): #ConnectToResource
        self.Trace("\t\t SendCTR")
        
        self.ifDll.SendCTR(self.scpFsmId, leg_id, ssfIndex)
        return
    
    def SendPNC(self, leg_id, ann_id, first_timeout, inter_timeout, ssfIndex=0): #PromptAndCollectUserInforation
        self.Trace("\t\t SendPNC")
        
        self.ifDll.SendPNC(self.scpFsmId, leg_id, ann_id, first_timeout, inter_timeout, ssfIndex)
        return
    
    def SendPNCWithArg(self, leg_id, pnc_arg, ssfIndex=0): # pnc_arg must be PTpnc_arg
        self.Trace("\t\t SendPNC")
        self.ifDll.SendPNCWithArg(self.scpFsmId, leg_id, pnc_arg, ssfIndex)
        
        return
    
    def SendPA(self, leg_id, pa_arg, ssfIndex=0): # pa_arg must be PTpa_arg
        self.Trace("\t\t SendPA")
        
        self.ifDll.SendPA(self.scpFsmId, leg_id, pa_arg, ssfIndex)
        return
    
    def SendPromptToCaller(self, annID):
        pa_arg = PTpa_arg()
        PTpa_arg_init(pa_arg)
        
        pa_arg.infoSend.annId.annType = ESRF_INFO_ANN_ID
        pa_arg.infoSend.annId.var.annId = annID        
        pa_arg.infoSend.numOfRepeat = 1
        
        self.SendPA(CALLER_LEGID, pa_arg)        
        return
        
    def SendMusicToCaller(self, duration=0, ssfIndex=0): #PlayAnnouncement.duration '0' means forever.
        self.Trace("\t\t SendMusicToCaller : duration=%d" % duration)
        
        self.ifDll.SendMusic(self.scpFsmId, CALLER_LEGID, ANN_ID_COMMON_MUSIC, duration, ssfIndex)
        return
    
    def SendRingBackToneToCaller(self, duration=0, ssfIndex=0): #PlayAnnouncement.duration '0' means forever.
        self.Trace("\t\t SendRingBackToneToCaller : duration=%d" % duration)
        
        self.ifDll.SendMusic(self.scpFsmId, CALLER_LEGID, ANN_ID_COMMON_RINGBACK, duration, ssfIndex)
        return
        

    def SendRecordANN(self, leg_id, record_ann_arg, ssfIndex=0):
        self.Trace("\t\t SendRecordAnn")
        self.ifDll.SendRecordANN(self.scpFsmId, leg_id, record_ann_arg,ssfIndex)
        return
    
    def SendDFC(self, leg_id, ssfIndex=0): #DisconnectForwardConnection
        self.Trace("\t\t SendDFC")

        self.ifDll.SendDFC(self.scpFsmId, leg_id, ssfIndex)
        return
            
    def SendConnect(self,dest_nbr, ssfIndex=0): #Connect
        self.Trace("\t\t SendConnect: dest_nbr=%s" % dest_nbr)
        
        self.ifDll.SendConnect(self.scpFsmId, dest_nbr, ssfIndex)
        return
    
    def SendConnectWithArg(self, conn_arg, ssfIndex=0): # conn_arg MUST be PTconnect_arg
        self.Trace("\t\t SendConnectWithArg")
        
        self.ifDll.SendConnectWithArg(self.scpFsmId, conn_arg, ssfIndex)
        return

    def SendICA(self, ica_arg, ssfIndex=0):
        self.Trace("\t\t SendICA")
        self.ifDll.SendICA(self.scpFsmId, ica_arg, ssfIndex)
        return
        
    def SendReconnect(self, ssfIndex=0): # Reconnect
        self.Trace("\t\t SendReconnect")
        self.ifDll.SendReconnect(self.scpFsmId, ssfIndex)
    
    def SendContinue(self, ssfIndex=0): #Continue
        self.Trace("\t\t SendContinue")
        
        self.ifDll.SendContinue(self.scpFsmId, ssfIndex)
        return
        
    def SendTransClose(self, ssfIndex=0): #Transaction close
        self.Trace("\t\t SendTransClose")
        
        self.ifDll.SendTransClose(self.scpFsmId, ssfIndex)
        return
    
    def SendReleaseCall(self, cause=16,ssfIndex=0): # ReleaseCall. Q.850 cause '16' is 'normal clear'.
        self.Trace("\t\t SendReleaseCall. cause=%d, ssfIndex=%d" % (cause, ssfIndex))
        
        self.ifDll.SendReleaseCall(self.scpFsmId, cause, ssfIndex)
        return

    def SendDisconnectLeg(self, legID, ssfIndex=0):
        self.Trace("\t\t SendDisconnectLeg")
        self.ifDll.SendDisconnectLeg(self.scpFsmId, legID, ssfIndex)
        return

    def SendJoinCall(self, peerSSFID, ssfIndex=0):
        self.Trace("\t\t SendJoinCall")
        self.ifDll.SendJoinCall(self.scpFsmId, peerSSFID, ssfIndex)
        return

    def SendBridge2Parties(self, b2p_arg, ssfIndex=0):
        self.Trace("\t\t SendBridge2Parties")
        self.ifDll.SendBridge2Parties(self.scpFsmId, b2p_arg, ssfIndex)
        return

    def SendPickUP(self, obcpID):
        self.Trace("\t\t SendPickUP")
        self.ifDll.SendPickUP(self.scpFsmId, obcpID)
        return

    def SendCallRetrieveReq(self, objScpID):
        self.Trace("\t\t SendCallRetrieveReq. objScpID=%08X" % objScpID)
        self.ifDll.PySendCallRetrieveReq(self.scpFsmId,objScpID)
        return

    def SendCallRetrievedResp(self, reqScpID):
        self.Trace("\t\t SendCallRetrievedResp")
        self.ifDll.PySendCallRetrieveResp(self.scpFsmId, reqScpID)
        return


    def SendUDPString(self, destIP, port, UdpStr):  #send UDP strings to destination IP address
        self.Trace("\t\t SendUDPString")

        self.ifDll.SendUDPString(destIP, port, UdpStr)
        return
        
    #=================================
    # system functions for MSS core
    #=================================   
    def getProcResult(self):
        return self.prcResult
        
    def getState(self):
        return self.state
    
    def setProcResult(self, result):
        self.prcResult = result['prcResult']
        return
    
    def setSysParams(self, params):
        self.scpFsmId = params['fsmID']
        self.profileID = params['profileID']
        self.foSSPInd = params['foSSPInd']
        self.tracedInd = params['tracedInd']
        return

    def setServiceInfoInd(self, infoInd):
        self.servInfoInd.freeCallInd = infoInd['freeCallInd']
        self.servInfoInd.suppressCallLevel = infoInd['suppressCallLevel']
        return

    #===================================================================
    #
    # message details from MSS core
    #
    #===================================================================
    def clearMessage(self):
        self.currMessage.clear()
        return

    def saveCommMsgInfo(self, msgParam):
        self.currMessage['msgType'] = msgParam['msgType']
        self.currMessage['orgTask'] = msgParam['orgTask']
        self.currMessage['orgFsmType'] = msgParam['orgFsmType']
        self.currMessage['orgFsmID'] = msgParam['orgFsmID']
        return

    def saveIDPMessage(self, idpParam): # InitialDP message
        self.currMessage['mediaDir'] = idpParam['mediaDir']
        self.currMessage['callerSrfID'] = idpParam['callerSrfID']
        self.currMessage['serviceNbr'] = idpParam['serviceNbr']

        self.callerNbr = idpParam['callerNbr']
        self.calledNbr = idpParam['calledNbr']
        self.redirNbr = idpParam['redirNbr']
        self.origDialPlan = idpParam['origDialPlan']
        self.ivrXMLServiceID = idpParam['ivrXMLServiceID']
        return

    def saveSRRMessage(self, msgParam):  # SpecialResourceReport
        self.currMessage['srrType'] = msgParam['srrType']
        return

    def setEDPInfos(self, edpInfo):
        self.edpEvent = edpInfo['edpEvent']
        self.edpLegId = edpInfo['edpLegId']
        self.currMessage['calledSrfID'] = edpInfo['calledSrfID']
        return

    def setSRFError(self, msgParam):
        self.srfError = msgParam['errType']
        return

    def setPNCResult(self, msgParam):
        self.pncResult = msgParam['digits']
        return

    #===================================================================
    #
    # Local user interfaces from MSS core
    #
    #===================================================================
    def getLocalUserPickUPGroupID(self):
        groupID = self.ifDll.PyGetLocalUserPickUPGroupID(self.profileID)
        return groupID


    def getLocalUserVoiceMailWelcomeAnnID(self):
        annID = self.ifDll.PyGetLocalUserVoiceMailWelcomeAnnID(self.profileID)
        return annID

    #===================================================================
    #
    # Core system parameters APIs
    #
    #===================================================================
    def getSysVarPickUpOnePrefix(self):
        prefixVal = create_string_buffer(64)
        self.ifDll.PyGetSysVarPickUPOnePrefix(prefixVal, 64)

        result = prefixVal.value
        return result


    def getSysVarVoiceMailMode(self):
        result = self.ifDll.PyGetSysVarVoiceMailMode()
        return result


    def getSysVarVoiceMailPrefixRecord(self):
        prefixVal = create_string_buffer(64)
        self.ifDll.PyGetSysVarVMRecordPrefix(prefixVal, 64)

        result = prefixVal.value
        return result


    def getSysVarVoiceMailPrefixRecordGreeting(self):
        prefixVal = create_string_buffer(64)
        self.ifDll.PyGetSysVarVMRcdGreetingPrefix(prefixVal, 64)

        result = prefixVal.value
        return result


    def getSysVarVoiceMailPrefixHearGreeting(self):
        prefixVal = create_string_buffer(64)
        self.ifDll.PyGetSysVarVMHearGreetingPrefix(prefixVal, 64)

        result = prefixVal.value
        return result


    def getSysVarVoiceMailPrefixHearVM(self):
        prefixVal = create_string_buffer(64)
        self.ifDll.PyGetSysVarVMHearVM(prefixVal, 64)

        result = prefixVal.value
        return result


    def getSysVarHGPrefixLogin(self):
        prefixVal = create_string_buffer(64)
        self.ifDll.PyGetSysVarHGOperLoginPrefix(prefixVal, 64)

        result = prefixVal.value
        return result


    def getSysVarHGPrefixLogout(self):
        prefixVal = create_string_buffer(64)
        self.ifDll.PyGetSysVarHGOperLogoutPrefix(prefixVal, 64)

        result = prefixVal.value
        return result


    def getSysVarCallParkPrefix(self):
        prefixVal = create_string_buffer(64)
        self.ifDll.PyGetSysVarCallParkPrefix(prefixVal, 64)

        result = prefixVal.value
        return result


    def getSysVarCallRetrievePrefix(self):
        prefixVal = create_string_buffer(64)
        self.ifDll.PyGetSysVarCallRetrievePrefix(prefixVal, 64)

        result = prefixVal.value
        return result

    #=================================
    # local locker functions
    #=================================
    
    # result: 0 - no; 1 - has
    def hasLocker(self, lockerName):
        result = self.ifDll.PyHasLocker(lockerName)
        if 1 == result:
            return MSS_PY_YES
            
        return MSS_PY_NO
        
    def setLocker(self, lockerName):
        self.ifDll.PySetLocker(lockerName)
        return
    
    def clrLocker(self, lockerName):
        self.ifDll.PyClrLocker(lockerName)
        return
        
    #=================================
    # Time functions
    #=================================    
    def getCurrSeconds(self):
        result = self.ifDll.PyGetCurrSeconds()
        return result    


    #=================================
    # local database functions
    #=================================
    def ldbQuery(self, sqlStr):  
        result = self.ifDll.PyLocalDBQuery(sqlStr)
        return result
    
    def ldbGetRowIntVal(self, rowIndex):
        result = self.ifDll.PyLocalDBGetRowIntVal(rowIndex)
        return result
    
    def ldbGetRowVal(self, rowIndex):
        rowBuf = create_string_buffer(256)
        self.ifDll.PyLocalDBGetRowVal(rowIndex, rowBuf, 255)

        result = rowBuf.value
        return result
    
    #=================================
    # cache database functions
    #=================================
    def cdbQuery(self, sqlStr):  
        result = self.ifDll.PyCacheDBQuery(sqlStr)
        return result
    
    def cdbGetRowIntVal(self, rowIndex):
        result = self.ifDll.PyCacheDBGetRowIntVal(rowIndex)
        return result
    
    def cdbGetRowVal(self, rowIndex):
        rowBuf = create_string_buffer(256)
        self.ifDll.PyCacheDBGetRowVal(rowIndex, rowBuf, 255)

        result = rowBuf.value
        return result
