"""
Spymemcached cache backend for django.

It is based on the original memcached backend.

Author: Flavio [FlaPer87] Percoco Premoli
"""

import sys

if not sys.platform.startswith("java"):
    raise RuntimeError("You need to use jython in order to use this backend")

import time
from django.core.cache.backends.base import BaseCache
from django.utils.encoding import smart_unicode, smart_str

try:
    from net.spy.memcached import *
    from net.spy.memcached.compat.SpyThread import *
except ImportError:
    raise ImportError("Spymemcached not found")

class CacheClass(BaseCache):
    def __init__(self, server, params):
        BaseCache.__init__(self, params)
        self._cache = MemcachedClient(AddrUtil.getAddresses(" ".join(server.split(';'))))

    def _get_memcache_timeout(self, timeout):
        """
        Memcached deals with long (> 30 days) timeouts in a special
        way. Call this function to obtain a safe value for your timeout.
        """
        timeout = timeout or self.default_timeout
        if timeout > 2592000: # 60*60*24*30, 30 days
            # See http://code.google.com/p/memcached/wiki/FAQ
            # "You can set expire times up to 30 days in the future. After that
            # memcached interprets it as a date, and will expire the item after
            # said date. This is a simple (but obscure) mechanic."
            #
            # This means that we have to switch to absolute timestamps.
            timeout += int(time.time())
        return timeout

    def add(self, key, value, timeout=0):
        if isinstance(value, unicode):
            value = value.encode('utf-8')
        return self._cache.add(smart_str(key), self._get_memcache_timeout(timeout), value)

    def get(self, key, default=None):
        val = self._cache.get(smart_str(key))
        if val is None:
            return default
        return val

    def set(self, key, value, timeout=0):
        self._cache.set(smart_str(key), self._get_memcache_timeout(timeout), value)

    def delete(self, key):
        self._cache.delete(smart_str(key))

    def get_many(self, keys):
        return self._cache.getBulk(map(smart_str,keys))

    def close(self, **kwargs):
        pass

    def incr(self, key, delta=1):
        # Needs tests
        try:
            val = self._cache.incr(key, delta)
        except ValueError:
            val = None
        if val is None:
            raise ValueError("Key '%s' not found" % key)

        return val

    def decr(self, key, delta=1):
        # Needs tests
        try:
            val = self._cache.decr(key, delta)
        except ValueError:
            val = None
        if val is None:
            raise ValueError("Key '%s' not found" % key)
        return val

    def set_many(self, data, timeout=0):
        safe_data = {}
        for key, value in data.items():
            if isinstance(value, unicode):
                value = value.encode('utf-8')
            self.set(key, value, timeout)

    def delete_many(self, keys):
        for k in keys:
            self.delete(k)

    def clear(self):
        self._cache.flush()
