#!python
# Copyright (c) 2015-2015 ActiveState Software Inc.
# See the file LICENSE.txt for licensing information.

from xpcom.components import interfaces as Ci
from xpcom.components import classes as Cc
from xpcom import components
import xml.etree.cElementTree as ET
import threading
import os.path
import time
import re
import apsw

import logging
import json

log = logging.getLogger("koScopeDocs-Py")
#log.setLevel(10)

prefs = Cc["@activestate.com/koPrefService;1"].getService(Ci.koIPrefService).prefs
koDirSvc  = Cc["@activestate.com/koDirs;1"].getService()
infoSvc = Cc["@activestate.com/koInfoService;1"].getService(Ci.koIInfoService)

catalogDir = os.path.join(koDirSvc.komodoPythonLibDir, "codeintel2", "catalogs")
stdlibDir = os.path.join(koDirSvc.komodoPythonLibDir, "codeintel2", "stdlibs")
stdlibUserDir = os.path.join(koDirSvc.userDataDir, "codeintel", "db", "stdlibs")

class koScopeDocs:
    
    _com_interfaces_ = [Ci.koIScopeDocs]
    _reg_desc_ = "Commando - Documentation Scope"
    _reg_clsid_ = "{b9f8948b-5b27-46eb-8787-839d6be59e86}"
    _reg_contractid_ = "@activestate.com/commando/koScopeDocs;1"
    
    searches = {}
    docs = {}
    
    indexing = False
    
    activeUuid = False
    
    @components.ProxyToMainThreadAsync
    def stopActiveSearches(self):
        for uuid in self.searches.keys():
            log.debug("Stopping search: " + uuid)
            self.searches[uuid].stop()
            del self.searches[uuid]

    def onSearch(self, query, name, uuid, onResults, onComplete):
        log.debug(uuid + " - Starting Search: " + query)
        
        self.activeUuid = uuid
        self.stopActiveSearches()
        
        query = query.strip().lower()
        
        if query == "" and name == "":
            catalogs = self.getCatalogs()
            for name in self.docs:
                if self.docs[name]["type"] == "catalog" and name not in catalogs:
                    continue
                
                self.callback(onResults, [{
                    "id": "docs-%s" % name,
                    "name": name,
                    "scope": "scope-docs",
                    "isScope": True
                }])
            self.callback(onComplete)
        else:
            self.searches[uuid] = Searcher(main=self, query=query, name=name, uuid=uuid, onResults=onResults, onComplete=onComplete)
            t = threading.Thread(target=self.searches[uuid].start, name="Docs Scope")
            t.setDaemon(True)
            t.start()
    
    def info(self, index, onComplete):
        conn = DB()
        
        row = conn.fetch("""
            SELECT * FROM entries
            JOIN docs ON
                entries.doc_id = docs.doc_id
            WHERE entry_id = ?
        """, (index,))
        
        if not row:
            self.callback(onComplete)
            return
        
        children = {}
        if row["children"]:
            for child in conn.fetchAll("SELECT * FROM entries WHERE entry_id IN (%s)" % row["children"]):
                if not child["type"]:
                    key = "properties"
                else:
                    key = "%ss" % child["type"]
                if not key in children:
                    children[key] = []
                children[key].append(child)
        row["children"] = children
        
        parents = [{"name": row["doc_name"], "index": 0}]
        if row["parents"]:
            for parent in row["parents"].split(","):
                [i,k] = parent.split("|")
                parents.append({"name": k, "index": i})
        row["parents"] = parents
        
        self.callback(onComplete, row)
        
        conn.close()
        
    def register(self, name, opts):
        opts = json.loads(opts)
        self.docs[name] = opts
    
    def getCatalogs(self):
        catalogs = prefs.getString('codeintel_selected_catalogs', '')
        if catalogs == "":
            catalogs = []
        else:
            catalogs = eval(catalogs)
            
        return catalogs
    
    def preload(self):
        t = threading.Thread(target=self._preload, name="Index Builder")
        t.setDaemon(True)
        t.start()
        
    def _preload(self):
        self.indexing = True
        
        log.debug("Building Databases from CIX files")
        
        # Regenerate the database if the build number or platform has changed
        platVersion = infoSvc.buildPlatform + infoSvc.buildNumber;
        dbpath = os.path.join(koDirSvc.userDataDir, "docs.db")
        if prefs.getString("docs_db_version","") != platVersion and os.path.isfile(dbpath):
            log.info("Re-generating the documentation database")
            try:
                os.remove(dbpath)
            except IOError:
                log.error("Removing the old documentation database failed, stop Komodo and delete it manually: " + dbpath)
            prefs.setString("docs_db_version", platVersion)
        
        indexer = Indexer(main=self)
        for name in self.docs:
            path = self._cixPathFor(name)
            if not path:
                log.error("Could not find path for " + name)
            else:
                indexer.build(name=name, path=path)
                
        indexer.conn.close()
                
        log.debug("Done Building Databases from CIX files")
        
        self.indexing = False
        
    def _cixPathFor(self, docName):
        key = self.docs[docName]["key"]
        kind = self.docs[docName]["type"]
        
        if kind != "stdlib":
            return os.path.join(catalogDir, key + ".cix")
        
        if os.path.isfile(os.path.join(stdlibDir, key + ".cix")):
            return os.path.join(stdlibDir, key + ".cix")
        
        result = None
        version = 0
        rx = re.compile(r'[^\d]+')
        for filename in os.listdir(stdlibDir):
            if filename.startswith("%s-" % key):
                _version = rx.sub('', filename)
                if _version > version:
                    result = os.path.join(stdlibDir, filename)
                    version = _version
        
        return result
        
    @components.ProxyToMainThreadAsync
    def callback(self, cb, value = "", code = 0):
        if not cb:
            return
        
        try:
            cb.callback(code, json.dumps(value))
        except Exception as e:
            #log.debug(value)
            log.exception(e)

class DB:
    
    def __init__(self):
        dbpath = os.path.join(koDirSvc.userDataDir, "docs.db")
        
        self.conn = apsw.Connection(dbpath)
        self.cursor = self.conn.cursor()
        self.open = True
        self.inTransaction = False
        
    @property
    def lastrowid(self):
        return self.conn.last_insert_rowid()
        
    def query(self, query, args = None):
        if args:
            return self.cursor.execute(query, args)
        else:
            return self.cursor.execute(query)
        
    def fetch(self, query, args = None):
        if self.inTransaction:
            self.commit()
            
        self.query(query, args)
        return self._row(self.cursor.fetchone())
    
    def fetchAll(self, query, args = None):
        if self.inTransaction:
            self.commit()
            
        for row in self.query(query, args):
            yield self._row(row)
            
    def commit(self):           
        log.debug("commit")
        self.inTransaction = False
        try:
            self.cursor.execute("commit")
        except apsw.SQLError:
            pass
            
    def close(self):
        self.commit()
        self.conn.close()
        self.open = False
       
    def transaction(self, query, args):
        if not self.inTransaction:
            self.query("BEGIN")
            self.inTransaction = True
            
        self.query(query, args)
    
    def _row(self, row):
        if not row:
            return None
        
        d = {}
        for idx, col in enumerate(self.cursor.description):
            d[col[0]] = row[idx]
        
        return d

class Searcher:
    
    types = ["class", "function", "constant", "property", "interface", "blob"]
    
    def __init__(self, main, query, name, uuid, onResults, onComplete):
        log.debug(uuid + " - Initializing Searcher")
        
        self.main = main
        self.query = query
        self.queryWords = re.split("\W+", query)
        self.name = name
        self.uuid = uuid
        self._onResults = onResults
        self._onComplete = onComplete
        
        self.resultsPending = []
        self.resultTimer = None
        self.completeTimer = None
        
        self._stop = False
        self.conn = False
        
    def start(self):
        while self.main.indexing:
            if self._stop:
                return
            
            time.sleep(1)
        
        self.conn = DB()
        
        clause = ""
        if self.name:
           key = self.main.docs[self.name]["key"]
           clause = "WHERE entry_search.doc_key='%s'" % key
        
        num = 0   
        maxResults = prefs.getLong("commando_search_max_results")
        
        if self.query:
            _clause = "* NEAR ".join(self.queryWords)
            _clause = _clause.replace("'","\\'")
            _clause = "%s*" % _clause
            
            if not clause:
                clause = "WHERE "
            else:
                clause = "%s AND " % clause
                
            clause = "%s entry_search.matchable MATCH '%s'" % (clause, _clause)
        
        query = """
            SELECT * FROM entry_search
                JOIN entries ON entry_search.entry_id = entries.entry_id
                JOIN docs ON entry_search.doc_id = docs.doc_id
                %s
            """ % clause
            
        columns = None
        for row in self.conn.fetchAll(query):
            if self._stop:
                return
            
            num = num + 1
            if num == maxResults:
                self.stop()
                return
            
            self.process(row)
        
        self.stop()
        
    def process(self, data):
        name = data["name"]
        
        kind = data["type"]
        if kind not in self.types:
            return
        
        matchable = data["matchable"]
        
        weight = 0
        if matchable == self.query:
            weight = weight + 50
        elif matchable.startswith(self.query):
            weight = weight + 25
            
        if name == self.query:
            weight = weight + 50
        elif name.startswith(self.query):
            weight = weight + 25
        
        # Prioritize results that match a larger part of the result
        if len(self.query) > 0:
            mx = len(matchable)
            diff = mx - len(self.query)
            weight = weight + (((mx - diff) * 25) / mx)
            
        parents = []
        if not self.name:
            parents.append(data["doc_name"])
        
        if data["parents"]:
            for parent in data["parents"].split(","):
                [i,k] = parent.split("|")
                parents.append(k)
            
        self.returnResult({
            "id": "doc-%s" % str(data["entry_id"]),
            "name": name,
            "description": "/".join(parents),
            "icon": "chrome://komodo/skin/images/codeintel/cb_%s.svg" % kind,
            "scope": "scope-docs",
            "weight": weight,
            "multiline": True,
            "data": {
                "index": str(data["entry_id"])
            }
        })
        
    def stop(self):
        self._stop = True
            
        if self.conn and self.conn.open:
            try:
                self.conn.close()
            except:
                # This usually means the connection was made on another thread
                # that thread will close the connection when it notices that
                # self._stop was set
                pass 
            
        self.onComplete()
        
    def onComplete(self):
        self.completeTimer = threading.Timer(0.06, self.onCompleteTimer)
        self.completeTimer.start()
            
    def onCompleteTimer(self):
        self.completeTimer = None
        self.main.callback(self._onComplete)
                
    def returnResult(self, result):
        if self._stop:
            return
        
        self.resultsPending.append(result)

        if not self.resultTimer:
            self.resultTimer = threading.Timer(0.05, self._returnResults)
            self.resultTimer.start()
            
    def _returnResults(self):
        self.resultTimer = None
        
        results = self.resultsPending
        self.resultsPending = []
        
        if self.uuid != self.main.activeUuid:
            return
        
        log.debug(self.uuid + " - Returning " + str(len(results)) + " results")
        self.main.callback(self._onResults, results)
    
class Indexer:
    conn = None
    ilkTypes = ["class", "function", "constant", "variable", "interface"]
    
    def __init__(self, main):
        log.debug("Initializing indexer")
        
        self.main = main
        self.idx = 1
        self.map = {}
        
        self.parentMap = {}
        self.idxtoElem = {}
        self.elemToIdx = {}
        
        if not self.conn:
            log.debug("Initializing db connection")
            dbpath = os.path.join(koDirSvc.userDataDir, "docs.db")
            exists = os.path.isfile(dbpath)
            
            self.conn = DB()
            
            if not exists:
                self.createDb()
        
        entry = self.conn.fetch('''
            SELECT MAX(entry_id) AS max FROM entries
        ''')
        
        if entry and entry["max"]:
            self.idx = entry["max"] + 1
            
        log.debug("Finished Initializing indexer")
            
    def createDb(self):
        log.debug("Creating database structure")
        
        self.conn.query('''
            CREATE TABLE entries (
                "entry_id" INTEGER PRIMARY KEY,
                "doc_id" INTEGER NOT NULL,
                "name" TEXT NOT NULL,
                "matchable" TEXT NOT NULL,
                "type" TEXT,
                "signature" TEXT,
                "returns" TEXT,
                "doc" TEXT,
                "parents" TEXT,
                "children" TEXT
            )
        ''')
        
        self.conn.query('''CREATE UNIQUE INDEX "entry_id" on entries (entry_id ASC)''')
        self.conn.query('''CREATE INDEX "entry_doc_id" on entries (doc_id ASC)''')
        
        self.conn.query('''
            CREATE TABLE docs (
                "doc_id" INTEGER PRIMARY KEY,
                "doc_name" TEXT NOT NULL,
                "doc_key" TEXT NOT NULL
            )
        ''')
        
        self.conn.query('''CREATE UNIQUE INDEX "doc_id" on docs (doc_id ASC)''')
        
        self.conn.query('''
            CREATE VIRTUAL TABLE entry_search
            USING fts4(entry_id, doc_id, doc_key, matchable)
        ''')
        
        self.conn.commit()
        
    def build(self, name, path, rebuild = False):
        log.debug("Building DB for " + path)
        
        doc = self.main.docs[name]
        key = doc["key"]
        kind = doc["type"]
        
        # Check for existing docs entry
        docEntry = self.conn.fetch('''
            SELECT * FROM docs WHERE doc_key=?
        ''', (key,))
        if not docEntry:
            self.conn.query('''
                INSERT INTO docs VALUES (NULL, ?, ?)
            ''', (name, key))
            docId = self.conn.lastrowid
        else:
            docId = docEntry["doc_id"]
            
        # Check if entries already exist for this doc
        docEntry = self.conn.fetch('''
            SELECT * FROM entries WHERE doc_id=?
        ''', (docId,))
        if docEntry:
            if not rebuild:
                log.debug("DB already exists, skipping")
                return
            else:
                # Delete existing entries
                self.conn.query('''DELETE FROM entries WHERE doc_id=?''', (docId,))
                self.conn.query('''DELETE FROM entry_search WHERE doc_id=?''', (docId,))
        
        if not os.path.isfile(path):
            log.error("Could not find cix for " + key)
            return
        
        # Build DOM cache
        log.debug("Building index for %s" % key)
        
        dom = {"tree": ET.ElementTree(file=path), "map": {}, "index": {} }
        acceptedTags = ["scope", "variable"]
        
        idx = self.idx
        for parent in dom["tree"].iter():
            if parent.tag not in acceptedTags:
                continue
            
            idx = idx + 1
            self.idxtoElem[idx] = parent
            self.elemToIdx[parent] = idx
            
            for child in parent:
                if parent.tag not in acceptedTags:
                    continue
                self.parentMap[child] = idx
        
        # Map the elements, assign them an index and record their parent
        for elem in dom["tree"].iter():
            if elem.tag not in acceptedTags:
                continue
            
            if not elem.get("name"):
                continue
            
            self.insert(elem, docId, doc)
            
        self.conn.commit()
        
        self.conn.query('''
            INSERT INTO entry_search
                SELECT entry_id, entries.doc_id, doc_key, matchable
                FROM entries JOIN docs ON entries.doc_id=docs.doc_id
                WHERE entries.doc_id=?
        ''', (str(docId),))
        
        self.conn.commit()
        
        self.idx = idx
        log.debug("Done building DB for " + path)
        
    def insert(self, elem, docId, doc):
        idx = self.elemToIdx[elem]
        
        matchable = []
        parents = []
        child = elem
        
        _parents = self.parents(elem, doc)
        for parent in _parents:
            pidx = self.elemToIdx[parent]
            parents.append('%s|%s' % (pidx, parent.get("name")))
            matchable.append(parent.get("name"))
        matchable.append(elem.get("name"))
        parents = ",".join(parents)
        matchable = " ".join(matchable)
        
        parent = None
        if elem in self.parentMap:
            pidx = self.parentMap[elem]
            parent = self.idxtoElem[pidx]
        
        children = []
        for child in elem:
            if child not in self.elemToIdx:
                continue
            cidx = self.elemToIdx[child]
            children.append(str(cidx))
        children = ",".join(children)
        
        kind = elem.get("ilk")
        if not kind and elem.tag == "variable":
            if parent and (parent.get("ilk") == "class" or parent.tag == "variable"):
                kind = "property"
            else:
                kind = "variable"
        
        self.conn.transaction('''
            INSERT INTO entries VALUES (?,?,?,?,?,?,?,?,?,?)
        ''', (idx,
              docId,
              elem.get("name"),
              matchable,
              kind,
              elem.get("signature"),
              elem.get("returns"),
              elem.get("doc"),
              parents,
              children))
        
    def parents(self, elem, doc):
        parents = []
        if elem not in self.parentMap:
            return parents
        
        pidx = self.parentMap[elem]
        parent = self.idxtoElem[pidx]
        while (parent and parent.tag in ["scope", "variable"]):
            parents.insert(0, parent)
            if parent in self.parentMap:
                parent = self.idxtoElem[self.parentMap[parent]]
            else:
                parent = False
            
        _parents = []
        namespace = []
        if "namespace" in doc:
            namespace = doc["namespace"]
        for index, elem in enumerate(parents):
            if index >= len(namespace) or elem.get("name") != namespace[index]:
                _parents.append(elem)
            
        return _parents
    