#include "ldraw.hpp"


// Class Point
	Point::Point()
	{
		//printf("constructeur point\n");
		m_x=0;
		m_y=0;
		m_z=0;
	}

	Point::Point(const Point &p)
	{
		m_x=p.m_x;
		m_y=p.m_y;
		m_z=p.m_z;
	}

	Point::Point(double x, double y, double z )
	{
		m_x=x;
		m_y=y;
		m_z=z;
	}

	Point::addp(Point* p)
	{
		p->m_x += m_x;
		p->m_y += m_y;
		p->m_z += m_z;
	}

	void Point::setp (double xi, double yi, double zi)
	{
		m_x=xi;
		m_y=yi;
		m_z=zi;
	}
	void Point::setp (const Point &p)
	{
		m_x=p.m_x;
		m_y=p.m_y;
		m_z=p.m_z;
	}

	char* Point::pparse (char* str)
	{
		char *p;
		p=str;
		if(sscanf(p,"%lf", &m_x) == 0) return NULL;
		// eat front spaces
		while(isspace(*p)!=0) p++;
		// find next space
		while(isspace(*p)==0) p++;
		if(sscanf(p,"%lf", &m_y) == 0) return NULL;
		// eat front spaces
		while(isspace(*p)!=0) p++;
		// find next space
		while(isspace(*p)==0) p++;
		if(sscanf(p,"%lf", &m_z) == 0) return NULL;
		// eat front spaces
		while(isspace(*p)!=0) p++;
		// find next space
		while(isspace(*p)==0) p++;
		return p;
	}

	double Point::distance (Point &p)
	{
		// Euclidian distance
		// return sqrt((p.m_x-m_x)*(p.m_x-m_x) + (p.m_y-m_y)*(p.m_y-m_y) + (p.m_z-m_z)*(p.m_z-m_z));
		// Manhattan distance
		return fabs(p.m_x-m_x) + fabs(p.m_y-m_y) + fabs(p.m_z-m_z);
	}

	// returns true if two points are close enough
	bool Point::compare(Point &p)
	{
		CompPoint++;
		if(fabs(p.m_x-m_x)>LdPrecision) return false;
		if(fabs(p.m_y-m_y)>LdPrecision) return false;
		if(fabs(p.m_z-m_z)>LdPrecision) return false;
		return true;
	}
	
	string Point::toString()
	{
		char buf[100];
		sprintf(buf,"%g %g %g", m_x, m_y, m_z);
		return buf;
	}

	void Point::print(bool lf) // parametre par defaut
	{

		cout << "x " << m_x << " y " << m_y << " z " << m_z;
// Version utilisant la surcharge. this est un pointeur -> on utilise *this
//		cout << *this;
//		if(lf) cout << endl;
	}
	void Point::printclassname()
	{
		cout << "classe point" << endl;
	}

	
// Surcharge de << pour pouvoir imprimer des points directement.
/*
ostream& operator << (ostream &s, Point &p)
{
	return s << p.m_x << ", " << p.m_y << ", " << p.m_z;
}

*/
//  == overload to compare "close enough" points
bool operator == (Point &p1, Point &p2)
{
	return (p1.compare(p2));
//	return (p1.distance(p2)<LdPrecision);
}

// Class Vector
/*
Vector::Vector(Point &p)
{
	m_x = p.getx();
	m_y = p.gety();
	m_z = p.getz();
}

Vector::Vector(double x, double y, double z )
{
	m_x = x;
	m_y = y;
	m_z = z;
}
*/
Vector::Vector(Point &p1, Point &p2)
{
	m_x = p2.getx() - p1.getx();
	m_y = p2.gety() - p1.gety();
	m_z = p2.getz() - p1.getz();
}

Vector* Vector::cross (Vector &v)
{
	double x,y,z;
	x = m_y*v.m_z - m_z*v.m_y;
	y = m_z*v.m_x - m_x*v.m_z;
	z = m_x*v.m_y - m_y*v.m_x;
	return new Vector(x,y,z);
}

double Vector::dot (Vector &v)
{
	return ( m_x*v.m_x + m_y*v.m_y + m_z*v.m_z );
}

double Vector::norm ()
{
	return sqrt( m_x*m_x + m_y*m_y + m_z*m_z );
}

void Vector::normalize ()
{
	double length;
	length=norm();
	if (length < EPSILON) return;
	m_x /= length;
	m_y /= length;
	m_z /= length;
}

// Class Matrix

Matrix::Matrix()
{
	m_mat[0][0]=0;
	m_mat[0][1]=0;
	m_mat[0][2]=0;
	m_mat[1][0]=0;
	m_mat[1][1]=0;
	m_mat[1][2]=0;
	m_mat[2][0]=0;
	m_mat[2][1]=0;
	m_mat[2][2]=0;
}

Matrix::Matrix(double a,double b,double c,double d,double e,double f,double g,double h,double i)
{
	m_mat[0][0]=a;
	m_mat[0][1]=b;
	m_mat[0][2]=c;
	m_mat[1][0]=d;
	m_mat[1][1]=e;
	m_mat[1][2]=f;
	m_mat[2][0]=g;
	m_mat[2][1]=h;
	m_mat[2][2]=i;
}

Matrix::Matrix(Point &p1, Point &p2, Point &p3)
{
	m_mat[0][0]=p1.getx();
	m_mat[0][1]=p1.gety();
	m_mat[0][2]=p1.getz();
	m_mat[1][0]=p2.getx();
	m_mat[1][1]=p2.gety();
	m_mat[1][2]=p2.getz();
	m_mat[2][0]=p3.getx();
	m_mat[2][1]=p3.gety();
	m_mat[2][2]=p3.getz();
}

string Matrix::toString()
{
	char buf[100];
	sprintf(buf,"%g %g %g %g %g %g %g %g %g", m_mat[0][0], m_mat[0][1], m_mat[0][2], m_mat[1][0], m_mat[1][1], m_mat[1][2], m_mat[2][0], m_mat[2][1], m_mat[2][2]);
	return buf;
}

Vector* Matrix::Mult(Vector &v1)
{
	double x,y,z;
	x =  m_mat[0][0]*v1.getx() +  m_mat[0][1]*v1.gety() + m_mat[0][2]*v1.getz();
	y =  m_mat[1][0]*v1.getx() +  m_mat[1][1]*v1.gety() + m_mat[1][2]*v1.getz();
	z =  m_mat[2][0]*v1.getx() +  m_mat[2][1]*v1.gety() + m_mat[2][2]*v1.getz();
	return new Vector(x,y,z);
}

Matrix* Matrix::MatMult(Matrix &m1)
{
	Matrix* m=new Matrix(0,0,0,0,0,0,0,0,0);
	for(int i=0; i<3; i++)
	{
		for(int j=0; j<3; j++)
		{
			for(int k=0; k<3; k++)
			{
				m->m_mat[i][j] += m_mat[i][k] * m1.m_mat[k][j];
			}
		}
	}
	return m;
}

double Matrix::MatDeterminant()
{
	return m_mat[0][0]*m_mat[1][1]*m_mat[2][2] + m_mat[1][0]*m_mat[2][1]*m_mat[0][2] + m_mat[2][0]*m_mat[0][1]*m_mat[1][2]
		- m_mat[2][0]*m_mat[1][1]*m_mat[0][2] - m_mat[0][0]*m_mat[2][1]*m_mat[1][2] - m_mat[1][0]*m_mat[0][1]*m_mat[2][2];
}


// Class LDComment

LdComment::LdComment(const LdComment &ldc)
{
	m_cs=ldc.m_cs;
}

LdComment::LdComment(const string &str)
{
	m_cs=str;
}

LdComment::LdComment(const char *str)
{
	m_cs=str;
}

string LdComment::toString()
{
	string result = "0 ";
	result += m_cs;
	return result;
}

int LdComment::readbfc()
{
	char comment[1000];
	strcpy(comment, m_cs.c_str());
	for(int i=0; i<strlen(comment); i++) comment[i]=toupper(comment[i]);
	if(strstr(comment,"BFC") == NULL) return 0;
	if(strstr(comment,"INVERTNEXT") != NULL) return 1;
	if(strstr(comment,"CERTIFY") == NULL) return 0;
	if(strstr(comment,"CCW") != NULL) return 2;
	if(strstr(comment,"CW") != NULL) return 3;
	return 2;
}

// Class LDSubfile
// *** treated as comment !!!***

/*
LdSubfile::LdSubfile(const LdSubfile &ldc)
{
	m_cs=ldc.m_cs;
}


LdSubfile::LdSubfile(const string &str)
{
	m_cs=str;
}
*/

LdSubfile::LdSubfile(const char *str)
{
	char*p;
	Point p1,p2,p3;
	p=(char*)str;
	// eat front spaces
	while(isspace(*p)!=0) p++;
	// find next space after type
	while(isspace(*p)==0) p++;
	sscanf(p,"%d", &m_col);
	// skip color
	while(isspace(*p)!=0) p++;
	// find next space
	while(isspace(*p)==0) p++;
	// Read 1st point
	p=m_origin.pparse(p);
	p=p1.pparse(p);
	p=p2.pparse(p);
	p=p3.pparse(p);
	m_mat=Matrix(p1,p2,p3);
	while(isspace(*p)!=0) p++;
	m_file=p;
}

LdSubfile::translate(Point &p)
{
	p.addp(&m_origin);
}

LdSubfile::transform(Matrix &m)
{
	Vector* v1;
	v1=m.Mult((Vector)m_origin);
	m_origin=*v1;
	delete v1;
	Matrix* m2;
	m2=m.MatMult(m_mat);
	m_mat=*m2;
	delete m2;
}



string LdSubfile::toString()
{
	char color[20];
	string result = "1 ";
	sprintf(color,"%d ",m_col);
	result += color;
	result += m_origin.toString();
	result += " ";
	result += m_mat.toString();
	result += " ";
	result += m_file;
	return result;
}


// Class LdLine

LdLine::LdLine(const char *str)
{
	char*p;
	p=(char*)str;
	// eat front spaces
	while(isspace(*p)!=0) p++;
	// find next space after type
	while(isspace(*p)==0) p++;
	sscanf(p,"%d", &m_col);
	// skip color
	while(isspace(*p)!=0) p++;
	// find next space
	while(isspace(*p)==0) p++;
	// Read 1st point
	p=m_p1.pparse(p);
	p=m_p2.pparse(p);
}

LdLine::LdLine(const Point &p1, const Point &p2, const int col)
{
	m_p1=p1;
	m_p2=p2;
	m_col=col;
}

string LdLine::toString()
{
	char color[20];
	string result = "2 ";
	sprintf(color,"%d ",m_col);
	result += color;
	result += m_p1.toString();
	result += " ";
	result += m_p2.toString();
	return result;
}

LdLine::translate(Point &p)
{
	p.addp(&m_p1);
	p.addp(&m_p2);
}

LdLine::transform(Matrix &m)
{
	Vector* v1;
	v1=m.Mult((Vector)m_p1);
	m_p1=*v1;
	delete v1;
	v1=m.Mult((Vector)m_p2);
	m_p2=*v1;
	delete v1;
}

LdObject* LdLine::getFirstLine()
{
	return new LdLine(m_p1, m_p2);
}

bool LdLine::matchLine(LdObject &line)
{
/*	if(line.getType() != Line)
	{
		cout << "Trying to use matchline with a non-line" << endl;
		return false;
	}
*/	
	if((m_p1 == ((LdLine&)line).m_p1) && (m_p2 == ((LdLine&)line).m_p2)) return true;
	if((m_p2 == ((LdLine&)line).m_p1) && (m_p1 == ((LdLine&)line).m_p2)) return true;

/* // Following code is not more efficient
	if(m_p1 == ((LdLine&)line).m_p1)
	{
		if(m_p2 == ((LdLine&)line).m_p2) return true;
		return false;
	}
	else if(m_p2 == ((LdLine&)line).m_p1)
	{
		if(m_p1 == ((LdLine&)line).m_p2) return true;
		return false;
	}
*/
	return false;
}

bool LdLine::matchPoint(Point &p1)
{
	if((m_p1==p1) || (m_p2==p1)) return true;
	return false;
}

// Class LdTriangle

LdTriangle::LdTriangle(const char *str)
{
	char*p;
	p=(char*)str;
	// eat front spaces
	while(isspace(*p)!=0) p++;
	// find next space after type
	while(isspace(*p)==0) p++;
	sscanf(p,"%d", &m_col);
	// skip color
	while(isspace(*p)!=0) p++;
	// find next space
	while(isspace(*p)==0) p++;
	// Read 1st point
	p=m_p1.pparse(p);
	p=m_p2.pparse(p);
	p=m_p3.pparse(p);
}

string LdTriangle::toString()
{
	char color[20];
	string result = "3 ";
	sprintf(color,"%d ",m_col);
	result += color;
	result += m_p1.toString();
	result += " ";
	result += m_p2.toString();
	result += " ";
	result += m_p3.toString();
	return result;
}


LdTriangle::translate(Point &p)
{
	p.addp(&m_p1);
	p.addp(&m_p2);
	p.addp(&m_p3);
}

LdTriangle::transform(Matrix &m)
{
	Vector* v1;
	v1=m.Mult((Vector)m_p1);
	m_p1=*v1;
	delete v1;
	v1=m.Mult((Vector)m_p2);
	m_p2=*v1;
	delete v1;
	v1=m.Mult((Vector)m_p3);
	m_p3=*v1;
	delete v1;
}

LdObject* LdTriangle::getFirstLine()
{
	m_edgecpt=0;
	return new LdLine(m_p1,m_p2);
}

LdObject* LdTriangle::getNextLine()
{
	m_edgecpt++;
	switch (m_edgecpt)
	{
	case 1:
		return new LdLine(m_p2,m_p3);
	case 2:
		return new LdLine(m_p3,m_p1);
	default:
		return NULL;
	}
}

bool LdTriangle::matchLine(LdObject &line)
{
	if(line.matchPoint(m_p1))
	{
		if(line.matchPoint(m_p2)) return true;
		if(line.matchPoint(m_p3)) return true;
		return false;
	}
	if(line.matchPoint(m_p2) && line.matchPoint(m_p3)) return true;
	return false;
}

Vector* LdTriangle::getNormal()
{
	Vector v1(m_p1, m_p2);
	Vector v2(m_p2, m_p3);
	Vector *v3 = v1.cross(v2);
	v3->normalize();
	return v3;
}

double LdTriangle::getAngle(LdObject &surf, LdObject &line )
{
	double result;
	Vector *v1 = getNormal();
	Vector *v2 = surf.getNormal();
	Vector *v3 = v1->cross(*v2);
	result = v3->norm();
	Vector v4(*((LdLine&)line).getP1(), *((LdLine&)line).getP2());
	result = 180*asin(result)/PI;
	if(v4.dot(*v3) > 0) result=-result;
	if(surf.GetBFC() == -1) result=-result;
	delete v1;
	delete v2;
	delete v3;
	return result;
}

Point* LdTriangle::getOtherPoint (LdObject &line)
{
	if(!((LdLine&)line).matchPoint(m_p1)) return &m_p1;
	if(!((LdLine&)line).matchPoint(m_p2)) return &m_p2;
	return &m_p3;
}

void LdTriangle::InvertWinding()
{
	Point p;
	p = m_p1;
	m_p1 = m_p3;
	m_p3 = p;
}


// Class LdQuad

LdQuad::LdQuad(const char *str)
{
	char*p;
	p=(char*)str;
	// eat front spaces
	while(isspace(*p)!=0) p++;
	// find next space after type
	while(isspace(*p)==0) p++;
	sscanf(p,"%d", &m_col);
	// skip color
	while(isspace(*p)!=0) p++;
	// find next space
	while(isspace(*p)==0) p++;
	// Read 1st point
	p=m_p1.pparse(p);
	p=m_p2.pparse(p);
	p=m_p3.pparse(p);
	p=m_p4.pparse(p);
}

string LdQuad::toString()
{
	char color[20];
	string result = "4 ";
	sprintf(color,"%d ",m_col);
	result += color;
	result += m_p1.toString();
	result += " ";
	result += m_p2.toString();
	result += " ";
	result += m_p3.toString();
	result += " ";
	result += m_p4.toString();
	return result;
}


LdQuad::translate(Point &p)
{
	p.addp(&m_p1);
	p.addp(&m_p2);
	p.addp(&m_p3);
	p.addp(&m_p4);
}

LdQuad::transform(Matrix &m)
{
	Vector* v1;
	v1=m.Mult((Vector)m_p1);
	m_p1=*v1;
	delete v1;
	v1=m.Mult((Vector)m_p2);
	m_p2=*v1;
	delete v1;
	v1=m.Mult((Vector)m_p3);
	m_p3=*v1;
	delete v1;
	v1=m.Mult((Vector)m_p4);
	m_p4=*v1;
	delete v1;
}


LdObject* LdQuad::getFirstLine()
{
	m_edgecpt=0;
	return new LdLine(m_p1,m_p2);
}

LdObject* LdQuad::getNextLine()
{
	m_edgecpt++;
	switch (m_edgecpt)
	{
	case 1:
		return new LdLine(m_p2,m_p3);
	case 2:
		return new LdLine(m_p3,m_p4);
	case 3:
		return new LdLine(m_p4,m_p1);
	default:
		return NULL;
	}
}

bool LdQuad::matchLine(LdObject &line)
{
	if(!line.matchPoint(m_p1) && !line.matchPoint(m_p3)) return false;
	LdLine ltemp (m_p1, m_p2);
	if(ltemp.matchLine(line)) return true;
	ltemp.SetLinePoints(m_p2, m_p3);
	if(ltemp.matchLine(line)) return true;
	ltemp.SetLinePoints(m_p3, m_p4);
	if(ltemp.matchLine(line)) return true;
	ltemp.SetLinePoints(m_p4, m_p1);
	if(ltemp.matchLine(line)) return true;
	return false;
}

Vector* LdQuad::getNormal()
{
	Vector v1(m_p1, m_p2);
	Vector v2(m_p2, m_p3);
	Vector *v3 = v1.cross(v2);
	v3->normalize();
	return v3;
}

double LdQuad::getAngle(LdObject &surf, LdObject &line )
{
	double result;
	Vector *v1 = getNormal();
	Vector *v2 = surf.getNormal();
	Vector *v3 = v1->cross(*v2);
	result = v3->norm();
	Vector v4(*((LdLine&)line).getP1(), *((LdLine&)line).getP2());
	if(v4.dot(*v3) > 0) result=-result;
	if(surf.GetBFC() == -1) result=-result;
	delete v1;
	delete v2;
	delete v3;
	return 180*asin(result)/PI;
}

Point* LdQuad::getOtherPoint (LdObject &line)
{
	if(!((LdLine&)line).matchPoint(m_p1)) return &m_p1;
	if(!((LdLine&)line).matchPoint(m_p2)) return &m_p2;
	return &m_p3;
}

void LdQuad::InvertWinding()
{
	Point p;
	p = m_p1;
	m_p1 = m_p3;
	m_p3 = p;
}



// Class LdCondline

LdCondline::LdCondline(const char *str)
{
	char*p;
	p=(char*)str;
	// eat front spaces
	while(isspace(*p)!=0) p++;
	// find next space after type
	while(isspace(*p)==0) p++;
	sscanf(p,"%d", &m_col);
	// skip color
	while(isspace(*p)!=0) p++;
	// find next space
	while(isspace(*p)==0) p++;
	// Read 1st point
	p=m_p1.pparse(p);
	p=m_p2.pparse(p);
	p=m_p3.pparse(p);
	p=m_p4.pparse(p);
}

LdCondline::LdCondline(Point &p1, Point &p2, Point &p3, Point &p4, const int col)
{
	m_p1=p1;
	m_p2=p2;
	m_p3=p3;
	m_p4=p4;
	m_col=col;
}


string LdCondline::toString()
{
	char color[20];
	string result = "5 ";
	sprintf(color,"%d ",m_col);
	result += color;
	result += m_p1.toString();
	result += " ";
	result += m_p2.toString();
	result += " ";
	result += m_p3.toString();
	result += " ";
	result += m_p4.toString();
	return result;
}


LdCondline::translate(Point &p)
{
	p.addp(&m_p1);
	p.addp(&m_p2);
	p.addp(&m_p3);
	p.addp(&m_p4);
}

LdCondline::transform(Matrix &m)
{
	Vector* v1;
	v1=m.Mult((Vector)m_p1);
	m_p1=*v1;
	delete v1;
	v1=m.Mult((Vector)m_p2);
	m_p2=*v1;
	delete v1;
	v1=m.Mult((Vector)m_p3);
	m_p3=*v1;
	delete v1;
	v1=m.Mult((Vector)m_p4);
	m_p4=*v1;
	delete v1;
}


LdObject* LdCondline::getFirstLine()
{
	return new LdLine(m_p1,m_p2);
}

bool LdCondline::matchLine(LdObject &line)
{
	LdLine ltemp (m_p1, m_p2);
	if(ltemp.matchLine(line)) return true;
	return false;
}

// Class LdPath
LdPath::LdPath()
{
/*	string* path;
	path=new string(".\\");
	pathtable.push_back(path);
	path=new string(".\\s\\");
	pathtable.push_back(path);
	path=new string(".\\p\\");
	pathtable.push_back(path);
	ifstream ldrawini("ldraw.ini", ios::in);
	if(ldrawini == NULL) cout << "ldraw.ini not found" << endl;
*/
}

LdPath::~LdPath()
{
	for(int i=0; i<pathtable.size(); i++)
	{
		delete pathtable[i];
	}
}

LdPath::addPath(string &s)
{
	string* path;
	path=new string(s);
	pathtable.push_back(path);
}

bool LdPath::subOpen(ifstream &ldSubfile, string &subfilename)
{
	for(int i=0; i<pathtable.size(); i++)
	{
		string sfn = *pathtable[i];
		sfn += subfilename;
		// cout << sfn.c_str() << endl;
		ldSubfile.open(sfn.c_str(), ios::in);
		if(ldSubfile.good()) return true;
		ldSubfile.clear();
	}
  return false;
}

string LdPath::toString()
{
	string result;
	for(int i=0; i<pathtable.size(); i++)
	{
		result += *pathtable[i];
		if(i==pathtable.size()-1) continue;
		result += ", ";
	}
	return result;
}

// Class Fldraw

#define BUFLEN 1000

void funcdelete (LdObject* ldo) 
{
	delete ldo;
}

Fldraw::~Fldraw()
{
	for_each(ldtab.begin(), ldtab.end(), funcdelete);
}

bool Fldraw::ldLoad (char *fname)
{
	char buf[BUFLEN];
	int type;
	int numline=0;
	int numEdge=0;
	int numCond=0;
	int numTri=0;
	int numQuad=0;
	int numSub=0;
	int curBFC=0;
	int newBFC;
	
	// ouverture fichier	
	ifstream ldFileIn(fname, ios::in);
	if(ldFileIn == NULL) return false;

	while (!ldFileIn.eof())
	{
		ldFileIn.getline(buf, BUFLEN);
		if(ldFileIn.fail())
		{
			if(ldFileIn.eof()) break;
			ldFileIn.clear();
			continue;
		}

		if (strlen(buf)==0) continue;
		sscanf(buf,"%d ", &type);

		switch(type)
		{
		case Comment:
			char *p;
			p=strchr(buf,'0');
			if(strlen(p)<2)
				*p=0;
			else
				p+=2;
			ldtab.push_back(new LdComment(p));
			newBFC = ((LdComment*)ldtab.back())->readbfc();
			switch (newBFC)
			{
			case 2:
				curBFC = 1;
				break;
			case 3:
				curBFC = -1;
				break;
			default:
				break;
			}
			break;
		case Subfile:
			numSub++;

			ldtab.push_back(new LdSubfile(buf));
			break;
		case Line:
			numEdge++;
			if(DelEdge) break;
			ldtab.push_back(new LdLine(buf));
			break;
		case Triangle:
			numTri++;
			ldtab.push_back(new LdTriangle(buf));
			break;
		case Quad:
			numQuad++;
			ldtab.push_back(new LdQuad(buf));
			break;
		case Condline:
			numCond++;
			if(DelCond) break;
			ldtab.push_back(new LdCondline(buf));
			break;
		default:
			break;
		}
		if(ldtab.back()->getType() != Subfile)	ldtab.back()->SetBFC(curBFC);
		else ldtab.back()->SetBFC(1);
	}
	m_nLine = ldtab.size();
	cout << endl << m_nLine << " lines in file"<< endl;
	cout << numEdge << " Edge lines";
	if(DelEdge) cout << " (deleted)";
	cout << endl << numCond << " Conditional lines";
	if(DelCond) cout << " (deleted)";
	cout << endl << numTri << " Triangles"<< endl;
	cout << numQuad << " Quads"<< endl;
	cout << numSub << " Subfiles"<< endl;

	return true;
}

Fldraw::ldExpand(LdPath *path)
{
	char buf[BUFLEN];
	int type;
	int numline=0;
	int numEdge=0;
	int numCond=0;
	int numTri=0;
	int numQuad=0;
	int numSub=0;
	int curBFC=0;
	int newBFC;
	int invertBFC;
	int invertsub;
	Point *po;
	Matrix *mat;
	for(int i=0; i<ldtab.size(); i++)
	{
		ifstream ldSubfile;

		if(ldtab[i]->getType() != Subfile) continue;
		// cout << ldtab[i]->toString().c_str() << endl;
		po=((LdSubfile *)ldtab[i])->getOrigin();
		mat=((LdSubfile *)ldtab[i])->getMAtrix();
		if(! path->subOpen(ldSubfile, ((LdSubfile *)ldtab[i])->getName())) continue;
		
		invertBFC = ((LdSubfile *)ldtab[i])->GetBFC();
		if(i>0)
		{
			if((ldtab[i-1]->getType() == Comment) && ((LdComment *)ldtab[i-1])->readbfc() == 1)
				invertBFC *= -1;
		}
		invertsub=invertBFC;
		if(mat->MatDeterminant() < 0) invertBFC *= -1; 
		while (!ldSubfile.eof())
		{
			ldSubfile.getline(buf, BUFLEN);
			if(ldSubfile.fail())
			{
				if(ldSubfile.eof()) break;
				ldSubfile.clear();
				continue;
			}

			if (strlen(buf)==0) continue;
			type=Empty;
			sscanf(buf,"%d ", &type);
			switch(type)
			{
			case Comment:
				char *p;
				p=strchr(buf,'0');
				if(p==NULL) break;
				if(strlen(p)<2)
					*p=0;
				else
					p+=2;
				ldtab.push_back(new LdComment(p));
				newBFC = ((LdComment*)ldtab.back())->readbfc();
				switch (newBFC)
				{
				case 2:
					curBFC = 1;
					break;
				case 3:
					curBFC = -1;
					break;
				case 1:
					break;
				default:
					if(!SubComments)
					{
						delete ldtab.back();
						ldtab.back() = new LdEmpty;
					}
					break;
				}
				break;
			case Subfile:
				numSub++;
				ldtab.push_back(new LdSubfile(buf));
				break;
			case Line:
				numEdge++;
				// if(DelEdge) break;
				ldtab.push_back(new LdLine(buf));
				break;
			case Triangle:
				numTri++;
				ldtab.push_back(new LdTriangle(buf));
				if(invertBFC == -1) ((LdTriangle*)ldtab.back())->InvertWinding();
				break;
			case Quad:
				numQuad++;
				ldtab.push_back(new LdQuad(buf));
				if(invertBFC == -1) ((LdTriangle*)ldtab.back())->InvertWinding();
				break;
			case Condline:
				numCond++;
				// if(DelCond) break;
				ldtab.push_back(new LdCondline(buf));
				break;
			default:
				break;
			}
			if(type != Empty)
			{
				ldtab.back()->SetNoOut(true);
				ldtab.back()->transform(*mat);
				ldtab.back()->translate(*po);
				if(ldtab.back()->getType() != Subfile)	ldtab.back()->SetBFC(curBFC);
				else ldtab.back()->SetBFC(invertsub);
			}
		}
		ldSubfile.close();
		if(Inlined)
		{
			string line = "Inlined: ";
			line += ldtab[i]->toString();
			delete ldtab[i];
			if(SubComments)	ldtab[i]=new LdComment(line.c_str());
			else ldtab[i]=new LdEmpty;
			if(i>0)
			{
				if((ldtab[i-1]->getType() == Comment) && ((LdComment *)ldtab[i-1])->readbfc() == 1)
				{
					delete ldtab[i-1];
					ldtab[i-1]=new LdEmpty;
				}
			}
		}
	}
	cout << "\nInlined from subfiles:\n";
	cout << numEdge << " Edge lines";
	// if(DelEdge) cout << " (deleted)";
	cout << endl << numCond << " Conditional lines";
	// if(DelCond) cout << " (deleted)";
	cout << endl << numTri << " Triangles"<< endl;
	cout << numQuad << " Quads"<< endl;
	m_nLine = ldtab.size();
}

bool Fldraw::ldSave (char *fname)
{
	int i=0;
	ofstream ldFileOut(fname, ios::out);
	if(ldFileOut == NULL) return false;
	if(WriteNewOnly) i = m_nLine;
	if(!Inlined || SubComments)
	{
		for(; i<ldtab.size(); i++)
		{
			if(!Inlined)
			{
				if(ldtab[i]->GetNoOut()) continue;
			}
			if(ldtab[i]->toString().length() > 0)	ldFileOut << ldtab[i]->toString().c_str() << endl;
		}
	}
	else
	{
		int newBFC;
		int curBFC=0;
		int invertBFC = 1;

		for(; i<ldtab.size(); i++)
		{
			switch(ldtab[i]->getType())
			{
			case Comment:
				newBFC = ((LdComment*)ldtab[i])->readbfc();
				if(newBFC ==0) break;
				if(curBFC == 0) 
				{
					curBFC = newBFC;
					break;
				}
				if(curBFC != newBFC) invertBFC = -1; else invertBFC = 1;
				delete ldtab[i];
				ldtab[i] = new LdEmpty;
				break;
			case Triangle:
			case Quad:
				if(invertBFC == -1)	ldtab[i]->InvertWinding();
				break;
			}
			if(ldtab[i]->toString().length() > 0)	ldFileOut << ldtab[i]->toString().c_str() << endl;
		}
	}
	return true;

}

Fldraw::findOrphan()
{
	int OrphanNumber = 0;
//	cout << "Taille avant " << ldtab.size() << endl;
	ldtab.push_back(new LdComment("// Unmatched edges"));
	for(int i=0; i<m_nLine; i++)
	{
		if(!ldtab[i]->isSurface()) continue;
		LdLine *ldl = (LdLine *)ldtab[i]->getFirstLine();
		do
		{
			bool foundmatch=false;
			for(int j=0; j<m_nLine; j++)
			{
				if(j==i) continue;
				if(ldtab[j]->matchLine(*ldl))
				{
					foundmatch=true;
					break;
				}
			}
			if(!foundmatch)
			{
				OrphanNumber++;
				ldl->SetLineColor(4); // Orphan lines are red
				ldtab.push_back(ldl);
			}
			else delete ldl;

		} while ((ldl = (LdLine *)ldtab[i]->getNextLine())!=NULL);
	
	}
//	cout << "Taille apres " << ldtab.size() << endl;
	cout << endl << OrphanNumber << " Unmatched edges found" << endl;
}

Fldraw::findAdjacent()
{
	int addLines=0;
	int addCond=0;
	int addInBetween=0;
	int Curcolor;
//	cout << "Taille avant " << ldtab.size() << endl;
	ldtab.push_back(new LdComment("// Added lines/Condlines"));
	for(int i=0; i<m_nLine; i++)
	{
		if(!ldtab[i]->isSurface()) continue;
		LdLine *ldl = (LdLine *)ldtab[i]->getFirstLine();
		do
		{
			bool foundmatch=true;
			for(int j=i+1; j<m_nLine; j++)
			{
				if(ldtab[j]->matchLine(*ldl))
				{
					for(int k=0; k<m_nLine; k++)
					{
						if(!ldtab[k]->isLine()) continue;
						if(ldtab[k]->matchLine(*ldl))
						{
							foundmatch=false;
							break;
						}
					}
					if(foundmatch)
					{
						double angle=ldtab[i]->getAngle(*ldtab[j], (LdObject&)*ldl);
						// cout << endl << angle << " degrees" << endl;
						if(fabs(angle) < LdFlat) 
						{
							delete ldl; 
							break;
						}
						if (fabs(angle) > LdMinEdge)
						{
							// Added lines are pink or edge color
							if(DebugColors) Curcolor=13; else Curcolor = 24;
							addLines++;
							ldl->SetLineColor(Curcolor); 
							ldtab.push_back(ldl);
							break;
						}
						if (fabs(angle) < LdMaxCond)
						{
							// Added condlines are light blue or edge color
							if(angle * CxCv > 0) break; 
							if(DebugColors) Curcolor=9; else Curcolor = 24;
							addCond++;
							ldtab.push_back(new LdCondline(*(ldl->getP1()), *(ldl->getP2()), 
								*(ldtab[i]->getOtherPoint(*ldl)), *(ldtab[j]->getOtherPoint(*ldl)), Curcolor));
							delete ldl;
							break;
						}

						if(angle * CxCv > 0) break; 
						addInBetween++;
						ldl->SetLineColor(2); // Added in-between lines are bright green
						ldtab.push_back(ldl);
						ldtab.push_back(new LdCondline(*(ldl->getP1()), *(ldl->getP2()), 
							*(ldtab[i]->getOtherPoint(*ldl)), *(ldtab[j]->getOtherPoint(*ldl)), 2)); // Added condlines are green
						break;
					}
				}
			}
			if(!foundmatch) delete ldl;
		} while ((ldl = (LdLine *)ldtab[i]->getNextLine())!=NULL);
	}
//	cout << "Taille apres " << ldtab.size() << endl;
	cout << endl << addLines << " New edges found" << endl;
	cout << addCond << " New conditional lines found" << endl;
	cout << addInBetween << " New dubious lines/condlines found" << endl;
}



